package org.simantics.utils.ui.dialogs;

import java.util.Collection;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.simantics.utils.strings.AlphanumComparator;
import org.simantics.utils.ui.internal.Activator;


/**
 * @author Tuukka Lehtonen
 */
public class ListDialog<T> extends Dialog {

    private static final String DIALOG = "ListDialog"; //$NON-NLS-1$

    private final Collection<T> selectables;

    private final String        title;

    private final String        description;

    private StructuredViewer         viewer;

    private IDialogSettings     dialogBoundsSettings;

    public ListDialog(Shell parent, Collection<T> selectables, String title, String description) {
        super(parent);
        this.selectables = selectables;
        this.title = title;
        this.description = description;

        IDialogSettings settings = Activator.getDefault().getDialogSettings();
        dialogBoundsSettings = settings.getSection(DIALOG);
        if (dialogBoundsSettings == null)
            dialogBoundsSettings = settings.addNewSection(DIALOG);
    }

    @Override
    protected IDialogSettings getDialogBoundsSettings() {
        return dialogBoundsSettings;
    }

    @Override
    protected void configureShell(Shell newShell) {
        if (title != null) {
            newShell.setText(title);
        } else {
            newShell.setText("Select");
        }
        super.configureShell(newShell);
    }

    @Override
    protected int getShellStyle() {
        return super.getShellStyle() | SWT.RESIZE;
    }

    @Override
    protected Point getInitialSize() {
        Point defaultSize = getShell().computeSize(SWT.DEFAULT, SWT.DEFAULT, true);
        Point result = super.getInitialSize();
        if (defaultSize.equals(result))
            return new Point(500, 300);
        return result;
    }
    
    protected StructuredViewer createViewer(Composite composite) {
    	return new TableViewer(composite, SWT.V_SCROLL | SWT.H_SCROLL | SWT.BORDER | SWT.SINGLE | SWT.FULL_SELECTION);
    }

    @Override
    protected Control createDialogArea(Composite parent) {
        Composite composite = (Composite) super.createDialogArea(parent);

        if (description != null) {
            Label label = new Label(composite, 0);
            label.setText(description);
            label.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false));
        }

        viewer = createViewer(composite);
        viewer.getControl().setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
        viewer.setContentProvider(new ArrayContentProvider());
        viewer.setLabelProvider(new SelectableLabelProvider());
        viewer.setSorter(sorter);
        viewer.setInput(selectables);

        applyDialogFont(composite);
        return composite;
    }

    @Override
    protected void okPressed() {
        super.okPressed();
    }

    private String toLabel(T t) {
        return t.toString();
    }

    private final ViewerSorter sorter = new ViewerSorter() {
        @SuppressWarnings("unchecked")
        @Override
        public int compare(Viewer viewer, Object e1, Object e2) {
            return AlphanumComparator.CASE_INSENSITIVE_COMPARATOR.compare(toLabel((T) e1), toLabel((T) e2));
        }
    };

    class SelectableLabelProvider extends LabelProvider {
        @SuppressWarnings("unchecked")
        @Override
        public String getText(Object element) {
            return toLabel((T) element);
        }
    }

}
