/*******************************************************************************
 * Copyright (c) 2010, 2023 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - gitlab #1039 
 *******************************************************************************/
package org.simantics.browsing.ui.model.nodetypes;

import java.util.Collection;
import java.util.Collections;

import org.eclipse.core.runtime.Platform;
import org.osgi.framework.Bundle;
import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.common.NodeContextBuilder;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Statement;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.ui.selection.WorkbenchSelectionElement;
import org.simantics.viewpoint.ontology.ViewpointResource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SpecialNodeType implements NodeType {
    
    private static final Logger LOGGER = LoggerFactory.getLogger(SpecialNodeType.class);
    
    public Resource resource;
    Class<?> contentType;
    protected transient String debugName;
    
    public SpecialNodeType(Resource resource, Class<?> contentType) {
        this.resource = resource;
        this.contentType = contentType;
    }

    public static SpecialNodeType create(ReadGraph g, Resource r) throws DatabaseException {
        ViewpointResource vr = ViewpointResource.getInstance(g);
        String contentTypeName = g.getPossibleRelatedValue(r, vr.HasContentType);
        Class<?> contentType;
        if("Resource".equals(contentTypeName))
            contentType = Resource.class;
        else if("Statement".equals(contentTypeName))
            contentType = Statement.class;
        else if("Variable".equals(contentTypeName))
            contentType = Variable.class;
        else {
            contentType = Object.class;
            if(contentTypeName != null)
                try {
                    String bundleId = g.getPossibleRelatedValue(r, vr.HasBundle);
                    Bundle bundle = null;
                    if (bundleId != null) {
                        bundle = Platform.getBundle(bundleId);
                        if (bundle == null)
                            LOGGER.warn("Referenced bundle '" + bundleId + "' not found in platform.");
                    }
                    if (bundle != null)
                        contentType = bundle.loadClass(contentTypeName);
                    else
                        contentType = Class.forName(contentTypeName);
                } catch (ClassNotFoundException e) {
                    LOGGER.error("Unknown content type {} - {}", contentTypeName, e.getMessage());
                    if (LOGGER.isTraceEnabled())
                        LOGGER.trace("Unknown content type {}", contentTypeName, e);
                }
            else
                LOGGER.warn("Content type is NULL.");
        }
        var result = new SpecialNodeType(r, contentType);
        result.debugName = result.toString(g);
        return result;
    }

    @Override
    public NodeContext createNodeContext(ReadGraph graph, Object content)
            throws DatabaseException {
        if(contentType.isInstance(content))
            return NodeContextBuilder.buildWithData(KEY_SEQUENCE,
                    new Object[] {content, this}
            );
        else
            return null;
    }

    @Override
    public Class<?> getContentType() {
        return contentType;
    }

    @Override
    public int hashCode() {
        return resource.hashCode();
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        SpecialNodeType other = (SpecialNodeType) obj;
        return resource.equals(other.resource);
    }

    @Override
    public boolean inherits(ReadGraph graph, NodeType superType) {
        // Special node type does not support inheritance
        return equals(superType);
    }

    @Override
    public Collection<NodeType> getSuper(ReadGraph g) {
        return Collections.emptyList();
    }

    protected String toDebugString(ReadGraph graph) throws DatabaseException {
        //return "SpecialNodeType(" + NameUtils.getSafeName(graph, resource) + ", " + contentType + ")";
        return "SpecialNodeType(" + NameUtils.getURIOrSafeNameInternal(graph, resource, true) + ", " + contentType + ")";
    }

    @Override
    public String toString(ReadGraph graph) throws DatabaseException {
        if (debugName == null)
            debugName = toDebugString(graph);
        return toString();
    }

    @Override
    public String toString() {
        return debugName != null ? debugName : super.toString();
    }

    @Override
    public WorkbenchSelectionElement getWorkbenchSelectionElement(NodeContext context) {
        return null;
    }

}
