/*******************************************************************************
 *  Copyright (c) 2023 Association for Decentralized Information Management
 *  in Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.web.serializer;

import java.awt.Shape;
import java.awt.geom.Path2D;
import java.awt.geom.PathIterator;
import java.io.IOException;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import com.fasterxml.jackson.databind.ser.std.StdSerializer;

/**
 * @author Jussi Koskela
 * @since 1.57.0
 */
public class ShapeSerializer extends StdSerializer<Shape> {

	private static final long serialVersionUID = -2239328941687246672L;

	public ShapeSerializer() {
		this(null);
	}

	public ShapeSerializer(Class<Shape> object) {
		super(object);
	}

	@Override
	public void serialize(Shape shape, JsonGenerator jgen, SerializerProvider provider) throws IOException {
		Path2D path;
		if (shape instanceof Path2D) {
			path = (Path2D) shape;
		} else {
			path = new Path2D.Double(shape);
		}

		ArrayNode pathNode = new ArrayNode(JsonNodeFactory.instance);
		PathIterator iter = path.getPathIterator(null);
		double x0 = 0, y0 = 0;
		while (!iter.isDone()) {
			ArrayNode arr = pathNode.addArray();
			double c[] = new double[6];
			int i = iter.currentSegment(c);
			int count;
			switch (i) {
			case PathIterator.SEG_MOVETO:
				arr.add("M");
				x0 = c[0];
				y0 = c[1];
				count = 2;
				break;
			case PathIterator.SEG_LINETO:
				arr.add("L");
				count = 2;
				break;
			case PathIterator.SEG_QUADTO:
				arr.add("Q");
				count = 4;
				break;
			case PathIterator.SEG_CUBICTO:
				arr.add("C");
				count = 6;
				break;
			case PathIterator.SEG_CLOSE:
				arr.add("L");
				c[0] = x0;
				c[1] = y0;
				count = 2;
				break;
			default:
				throw new RuntimeException("Unexpected segment type " + i);
			}
			for (int j = 0; j < count; j++) {
				arr.add(c[j]);
			}
			iter.next();
		}

		jgen.writeRawValue(pathNode.toString());
	}

}