package org.simantics.logging;

import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.stream.Collectors;

import org.slf4j.LoggerFactory;

import ch.qos.logback.classic.Logger;
import ch.qos.logback.classic.LoggerContext;
import ch.qos.logback.classic.spi.ILoggingEvent;
import ch.qos.logback.core.Appender;
import ch.qos.logback.core.FileAppender;
import ch.qos.logback.core.rolling.RollingFileAppender;
import ch.qos.logback.core.spi.AppenderAttachable;

public class LogbackLogProvider implements LogProvider {

    private static final org.slf4j.Logger LOGGER = LoggerFactory.getLogger(LogbackLogProvider.class);
    
    @Override
    public List<Path> get() {
        List<Path> logs = new ArrayList<>();
        try {
            LoggerContext context = (LoggerContext) LoggerFactory.getILoggerFactory();
            Logger logger = context.getLogger(Logger.ROOT_LOGGER_NAME);
            Iterator<Appender<ILoggingEvent>> appenderIter = logger.iteratorForAppenders();
            while (appenderIter.hasNext()) {
                FileAppender<ILoggingEvent> appender = findFileAppender(appenderIter.next());
                if (appender != null) {
                    String logFile = ((FileAppender<ILoggingEvent>)appender).getFile();
                    Path log = Paths.get(logFile).toAbsolutePath();
                    if (appender instanceof RollingFileAppender) {
                        // Collect all logs
                        Path parent = log.getParent();
                        List<Path> newLogs = Files.walk(parent).collect(Collectors.toList());
                        if (LOGGER.isDebugEnabled())
                            LOGGER.debug("Found {} from {}", newLogs, appender);
                        logs.addAll(newLogs);
                    } else {
                        logs.add(log);
                    }
                } else {
                    if (LOGGER.isDebugEnabled()) {
                        LOGGER.debug("Appender is not {} but is {} instead", FileAppender.class.getName(), appender != null ? appender.getClass().getName() : "null");
                    }
                }
            }
        } catch (ClassCastException e) {
            // Okay, we are not using logback here
            if (LOGGER.isDebugEnabled())
                LOGGER.debug("Seems like we are not using logback but {} instead", LoggerFactory.getILoggerFactory(), e);
        } catch (Throwable t) {
            LOGGER.error("Could not collect logs", t);
        }
        if (LOGGER.isDebugEnabled())
            LOGGER.debug("Found {} log files : {}", logs.size(), logs);
        return logs;
    }
    
    private static FileAppender<ILoggingEvent> findFileAppender(Appender<ILoggingEvent> appender) {
        if (appender instanceof AppenderAttachable) {
            // Ok, has child appender
            Iterator<Appender<ILoggingEvent>> children = ((AppenderAttachable<ILoggingEvent>) appender).iteratorForAppenders();
            while (children.hasNext()) {
                FileAppender<ILoggingEvent> app = findFileAppender(children.next());
                // TODO: returns only first FileAppender that it founds but not a collection
                if (app != null)
                    return app;
            }
            return null;
        } else if (appender instanceof FileAppender) {
            return (FileAppender<ILoggingEvent>) appender;
        } else {
            return null;
        }
    }

}
