package org.simantics.logging;

import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.qos.logback.classic.Level;
import ch.qos.logback.classic.LoggerContext;

/**
 * Class for modifying the active logging configuration
 * 
 * @author Jani Simomaa
 *
 */
public final class LogConfigurator {

    private static final Logger LOGGER = LoggerFactory.getLogger(LogConfigurator.class);

    private LogConfigurator() {
    }

    /**
     * Sets logging level to represent the given argument
     * 
     * @param level ERROR WARN INFO DEBUG TRACE
     */
    public static void setLoggingLevel(String level) {
        if (LOGGER.isInfoEnabled())
            LOGGER.info("Setting logger level to {}", level);
        LoggerContext context = (LoggerContext) LoggerFactory.getILoggerFactory();
        Level ll = getLoggerLevel(level);
        List<ch.qos.logback.classic.Logger> loggerList = context.getLoggerList();
        loggerList.forEach(l -> l.setLevel(ll));
        if (LOGGER.isDebugEnabled())
            LOGGER.debug("Loggers installed {}", loggerList);
    }

    public static void setLoggingLevelForLogger(String logger, String level) {
        if (LOGGER.isInfoEnabled())
            LOGGER.info("Setting logger level to {} for loggers {}", level, logger);
        LoggerContext context = (LoggerContext) LoggerFactory.getILoggerFactory();
        Level ll = getLoggerLevel(level);
        ch.qos.logback.classic.Logger l = context.getLogger(logger);
        l.setLevel(ll);
        if (LOGGER.isDebugEnabled())
            LOGGER.debug("Logger {} level set to {}", l, ll);
    }

    private static Level getLoggerLevel(String level) {
        return Level.valueOf(level);
    }

    /**
     * @since 1.47.0
     */
    public static List<LoggerLevel> listConfiguredLoggers() {
        LoggerContext context = (LoggerContext) LoggerFactory.getILoggerFactory();
        return context.getLoggerList().stream()
                .map(l -> {
                    //LOGGER.info("{} : {} : {} : {}", l, l.getName(), l.getLevel(), l.getEffectiveLevel());
                    if (l.getName().isEmpty())
                        return null;
                    return new LoggerLevel(l, l.getEffectiveLevel().toString());
                })
                .filter(Objects::nonNull)
                .sorted()
                .collect(Collectors.toList());
    }

    /**
     * @since 1.47.0
     */
    public static void applyLogLevels(List<LoggerLevel> loggers) {
        loggers.forEach(l -> {
            Level level = getLoggerLevel(l.getLevel());
            Logger logger = l.getLogger();
            if (logger instanceof ch.qos.logback.classic.Logger) {
                LOGGER.info("Setting existing logger {} level to {}", l.getName(), l.getLevel());
                ch.qos.logback.classic.Logger ll = (ch.qos.logback.classic.Logger) l.getLogger();
                ll.setLevel(level);
            } else if (!l.getName().trim().isEmpty()) {
                LOGGER.info("Defining new logger {} with level {}", l.getName(), l.getLevel());
                logger = LoggerFactory.getLogger(l.getName());
                if (logger instanceof ch.qos.logback.classic.Logger) {
                    ch.qos.logback.classic.Logger ll = (ch.qos.logback.classic.Logger) logger;
                    ll.setLevel(level);
                }
            }
        });
    }

}
