/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.graph.store;

import gnu.trove.list.array.TIntArrayList;
import gnu.trove.map.hash.THashMap;
import gnu.trove.map.hash.TIntIntHashMap;
import gnu.trove.procedure.TIntProcedure;

import java.util.ArrayList;
import java.util.Collection;
import java.util.concurrent.Callable;

import org.simantics.graph.query.Path;
import org.simantics.graph.query.Res;
import org.simantics.graph.utils.GraphExecutor;

public class GraphStore {	
	public StatementStore statements;
	public IdentityStore identities;
	public ValueStore values;
	protected THashMap<Class<?>, IStore> stores;
	
	public GraphStore(StatementStore statements, IdentityStore identities, ValueStore values, THashMap<Class<?>, IStore> stores) {
		this.statements = statements;
		this.identities = identities;
		this.values = values;
		this.stores = stores;
	}
	
	public GraphStore() {
		this(new StatementStore(), new IdentityStore(), new ValueStore(), new THashMap<Class<?>, IStore>());	
	}
	
	public GraphStore(GraphStore store) {
		this(store.statements, store.identities, store.values, store.stores);
	}

	public void map(final TIntIntHashMap map) {
		ArrayList<Callable<Object>> tasks = new ArrayList<Callable<Object>>();
		tasks.add(new IStore.MapTask(statements, map));
		tasks.add(new IStore.MapTask(values, map));
		for(IStore store : stores.values())
			tasks.add(new IStore.MapTask(store, map));
		tasks.add(new IStore.MapTask(identities, map));
		
		try {
			GraphExecutor.EXECUTOR.invokeAll(tasks);
		} catch (InterruptedException e) {
			throw new RuntimeException(e);
		}
	}
	
	public <T extends IStore> void addStore(Class<T> clazz, T store) {
		stores.put(clazz, store);
	}
	
	@SuppressWarnings("unchecked")
	public <T extends IStore> T getStore(Class<T> clazz) {
		return (T)stores.get(clazz);
	}
	
	public int resToId(Res res) {
		if(res instanceof Path)
			return identities.pathToId((Path)res);
		else {
			IdRes idRes = (IdRes)res;
			if(idRes.fragment != this)
				return -1;
			else
				return idRes.id;
		}
	}
	
	public int createResToId(Res res) {
		if(res instanceof Path)
			return identities.createPathToId((Path)res);
		else {
			IdRes idRes = (IdRes)res;
			if(idRes.fragment != this)
				throw new RuntimeException("Cannot crate reference to an internal resource of other graph");
			else
				return idRes.id;
		}
	}
	
	public Res idToRes(int id) {
		Path path = identities.idToPath(id);
		if(path == null)
			return new IdRes(this, id);
		else
			return path;
	}
	
	public void addIdsToResult(TIntArrayList ids, final Collection<Res> result) {
		ids.forEach(new TIntProcedure() {
			
			@Override
			public boolean execute(int value) {
				result.add(idToRes(value));
				return true;
			}
		});
	}
	
	public void collectReferences(boolean[] set) {
		statements.collectReferences(set);
		identities.collectReferences(set);
		values.collectReferences(set);		
	}
	
	@Override
	public String toString() {
		final StringBuilder b = new StringBuilder();
		statements.forStatements(new IStatementProcedure() {
			
			public String name(int id) {
				Path path = identities.idToPath(id);
				if(path == null)
					return "(" + id + ")";
				else
					return path + "(" + id + ")";
			}
			
			@Override
			public void execute(int s, int p, int o) {
				b.append(name(s));
				b.append(" ");
				b.append(name(p));
				b.append(" ");
				b.append(name(o));
				b.append("\n");
			}
		});
		return b.toString();
	}	
}
