package org.simantics.filesystem.services.internal.sizetracker;

import java.io.IOException;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;

import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Deactivate;
import org.simantics.filesystem.services.sizetracker.DirectorySizeService;
import org.simantics.filesystem.services.sizetracker.SizeTracker;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Tuukka Lehtonen
 * @since 1.31.0
 */
@Component
public class DirectorySizeServiceImpl implements DirectorySizeService {

	private Logger logger = LoggerFactory.getLogger(DirectorySizeServiceImpl.class);

	private final Map<Path, SizeTrackerImpl> trackers = new HashMap<>();

	@Override
	public SizeTracker track(Path directory) throws IOException {
		synchronized (trackers) {
			SizeTrackerImpl tracker = trackers.get(directory);
			if (tracker != null)
				return tracker;
			tracker = new SizeTrackerImpl(this, directory);
			try {
				tracker.start();
				trackers.put(directory, tracker);
				return tracker;
			} catch (IOException e) {
				safeCloseTracker(tracker);
				throw e;
			}
		}
	}

	void removeTracker(Path path) {
		synchronized (trackers) {
			trackers.remove(path);
		}
	}

	public void shutdown() {
		synchronized (trackers) {
			trackers.forEach((p,t) -> safeCloseTracker(t));
			trackers.clear();
		}
	}

	private void safeCloseTracker(SizeTrackerImpl t) {
		try {
			t.close();
		} catch (IOException e) {
			logger.error("Error closing size tracker {}", t, e);
		}
	}

	@Activate
	public void activate() {
		//System.out.println("Activated DirectorySizeServiceImpl");
	}

	@Deactivate
	public void deactivate() {
		//System.out.println("Deactivated DirectorySizeServiceImpl");
		shutdown();
	}

}
