/*******************************************************************************
 * Copyright (c) 2007, 2023 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - GitLab #927
 *******************************************************************************/
package org.simantics.document.ui.graphfile;

import static java.nio.file.StandardWatchEventKinds.ENTRY_DELETE;
import static java.nio.file.StandardWatchEventKinds.ENTRY_MODIFY;
import static java.nio.file.StandardWatchEventKinds.OVERFLOW;

import java.io.File;
import java.io.IOException;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.WatchEvent;
import java.nio.file.WatchEvent.Kind;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicBoolean;

import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.preferences.DefaultScope;
import org.eclipse.core.runtime.preferences.IEclipsePreferences;
import org.eclipse.core.runtime.preferences.IEclipsePreferences.IPreferenceChangeListener;
import org.eclipse.core.runtime.preferences.InstanceScope;
import org.eclipse.ui.PartInitException;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.layer0.variable.Variables;
import org.simantics.document.DocumentResource;
import org.simantics.document.ui.Activator;
import org.simantics.document.ui.prefs.DocumentsPreferences;
import org.simantics.editors.Editors;
import org.simantics.graphfile.ontology.GraphFileResource;
import org.simantics.graphfile.util.GraphFileUtil;
import org.simantics.layer0.Layer0;
import org.simantics.ui.workbench.editor.AbstractResourceEditorAdapter;
import org.simantics.ui.workbench.editor.EditorAdapter;
import org.simantics.ui.workbench.editor.EditorRegistry;
import org.simantics.ui.workbench.editor.IEditorRegistry;
import org.simantics.utils.ui.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ExternalEditorAdapter extends AbstractResourceEditorAdapter implements EditorAdapter {

    private static final Logger LOGGER = LoggerFactory.getLogger(ExternalEditorAdapter.class);

    private static final Map<Path, Resource> tempFiles = new ConcurrentHashMap<>();
    private static ExternalFileWatcher fileWatcher;

    private static final int PREFERRED_PRIORITY = 1 << 30;
    private static final int MAX_FILENAME_LENGTH = 255;
    private static final int MAX_DIRNAME_LENGTH = 255;

    private IEclipsePreferences defaultPreferenceNode;
    private IEclipsePreferences instancePreferenceNode;
    private boolean isPreferredAdapter = false;

    private boolean getIsPreferredAdapterPreference() {
        boolean def = defaultPreferenceNode.getBoolean(
                DocumentsPreferences.P_PREFER_EXTERNAL_EDITOR_FOR_DOCUMENTS,
                DocumentsPreferences.DEFAULT_PREFER_EXTERNAL_EDITOR_FOR_DOCUMENTS);
        return instancePreferenceNode.getBoolean(
                DocumentsPreferences.P_PREFER_EXTERNAL_EDITOR_FOR_DOCUMENTS,
                def);
    }

    IPreferenceChangeListener preferenceListener = event -> {
        String key = event.getKey();
        if (key.equals(DocumentsPreferences.P_PREFER_EXTERNAL_EDITOR_FOR_DOCUMENTS)) {
            boolean preferred  = getIsPreferredAdapterPreference();
            if (preferred == isPreferredAdapter)
                return;

            isPreferredAdapter = preferred;

            // Ensure the registry doesn't contain any old cached
            // preferences to force it to re-resolve adapters by priority.
            IEditorRegistry er = EditorRegistry.getInstance();
            er.getMappings().clear();
            er.clearCache();
        }
    };

    public ExternalEditorAdapter() {
        super(Messages.ExternalEditorAdapter_ExternalEditor, Activator.imageDescriptorFromPlugin("com.famfamfam.silk", "icons/page.png")); //$NON-NLS-2$ //$NON-NLS-3$

        defaultPreferenceNode = DefaultScope.INSTANCE.getNode(Activator.PLUGIN_ID);
        instancePreferenceNode = InstanceScope.INSTANCE.getNode( Activator.PLUGIN_ID );
        isPreferredAdapter = getIsPreferredAdapterPreference();
        instancePreferenceNode.addPreferenceChangeListener( preferenceListener );
    }

    @Override
    public int getPriority() {
        if (isPreferredAdapter)
            return PREFERRED_PRIORITY;
        return super.getPriority();
    }

    @Override
    public void setPriority(int priority) {
        super.setPriority(priority);
    }

    @Override
    public boolean canHandle(ReadGraph g, Resource r) throws DatabaseException {
        DocumentResource doc = DocumentResource.getInstance(g);
        if (!g.isInstanceOf(r, doc.FileDocument))
            return false;
        return true;
    }

    @Override
    protected void openEditor(final Resource input) throws Exception {
        Path path = null;
        for (Entry<Path, Resource> entry : tempFiles.entrySet()) {
            Resource value = entry.getValue();
            if (input.equals(value)) {
                path = entry.getKey();
            }
        }
        if (path == null) {
            path = Simantics.getSession().syncRequest(new UniqueRead<Path>() {

                @Override
                public Path perform(ReadGraph graph) throws DatabaseException {
                    GraphFileResource GF = GraphFileResource.getInstance(graph);

                    File root = new File(Platform.getInstanceLocation().getURL().getPath(), "tempFiles/external"); //$NON-NLS-1$

                    Variable v = Variables.getPossibleVariable(graph, input);
                    String relativePath = null;

                    if (v != null) {
                        String externalFilePath = v.getPossiblePropertyValue(graph, GF.ExternalFilePath);
                        if (externalFilePath != null) {
                            relativePath = sanitizePath(new File(externalFilePath)).getPath();
                        }
                    } else {
                        String externalFilePath = graph.getPossibleRelatedValue(input, GF.ExternalFilePath);
                        if (externalFilePath != null) {
                            relativePath = sanitizePath(new File(externalFilePath)).getPath();
                        }
                    }
                    if (relativePath == null) {
                        relativePath = graph.getPossibleRelatedValue(input, Layer0.getInstance(graph).HasName).toString();
                        if (relativePath != null) {
                            relativePath = sanitizeName(relativePath, false);
                        } else {
                            relativePath = "Untitled"; //$NON-NLS-1$
                        }
                    }

                    File file = new File(root, relativePath);
                    String name = file.getName();
                    int pos = name.lastIndexOf('.'); //$NON-NLS-1$
                    String ext = "";
                    String base = name;
                    if (pos != -1) {
                        ext = name.substring(pos);
                        base = name.substring(0, pos);
                    }

                    File dir = file.getParentFile();
                    String shortenedName = shortenFileName(base, ext, 1);
                    file = new File(dir, shortenedName);

                    int i = 2;
                    while (file.exists()) {
                        shortenedName = shortenFileName(base, ext, i);
                        file = new File(dir, shortenedName); 
                    }

                    try {
                        dir.mkdirs();
                        watch(dir.toPath());

                        final File finalFile = file;
                        tempFiles.compute(finalFile.toPath(), (t,y) -> {
                            try {
                                GraphFileUtil.writeDataToFile(graph, input, t.toFile());
                            } catch (IOException | DatabaseException e) {
                                LOGGER.error("Could not write data to file ", e); //$NON-NLS-1$
                            }
                            return input;
                        });
                        LOGGER.info("Adding tempFile {} with resource {}", file, input); //$NON-NLS-1$
                        return file.toPath();
                    } catch (IOException e2) {
                        ExceptionUtils.logAndShowError(e2);
                        return null;
                    }
                }
            });
        }
        if (path != null) {
            try {
                Editors.openExternalEditor(path.toFile());
            } catch (PartInitException e) {
                ExceptionUtils.logAndShowError(e);
            }
        }
    }

    private static File sanitizePath(File f) {
        return sanitizePath(f, false);
    }
    
    private static File sanitizePath(File f, boolean isDir) {
        String fileName = sanitizeName(f.getName(), isDir);
        File parent = f.getParentFile();
        if (parent != null) {
            return new File(sanitizePath(parent, true), fileName);
        } else {
            return new File(fileName);
        }
    }
    
    private static String shortenFileName(String baseName, String ext, int variant) {
        String num = variant != 1 ? Integer.toString(variant) : "";
        String shortenedBase = baseName;
        if (baseName.length() + num.length() + ext.length() > MAX_FILENAME_LENGTH) {
            if (num.length() + ext.length() > MAX_FILENAME_LENGTH) ext = ext.substring(0, MAX_FILENAME_LENGTH - num.length()); // cut very long extension names
            shortenedBase = shortenedBase.substring(0, MAX_FILENAME_LENGTH - num.length() - ext.length()); // make sure that the name is not too long
        }
        return shortenedBase + ext  + num;
    }

    private static String sanitizeName(String name, boolean isDir) {
        name = name.replaceAll("(?i)^(con|prn|aux|nul|com[0-9]|lpt[0-9])(\\.[^.]*)?$", "$1_"); //$NON-NLS-1$ // append underscore to forbidden names
        name = name.replaceAll("[<>:\"\\/|?*\\x00-\\x1F]", "_"); //$NON-NLS-1$ // replace forbidden characters with underscore
        name = name.replaceAll("^(.*[\\. ])$", "$1_"); //$NON-NLS-1$ // append underscore if space or dot is last character
        if (isDir && name.length() > MAX_DIRNAME_LENGTH) name = name.substring(0, MAX_DIRNAME_LENGTH - 1) + "_"; //$NON-NLS-1$
        return name;
    }

    private static void watch(Path path) throws IOException {
        synchronized (ExternalEditorAdapter.class) {
            if (fileWatcher == null) {
                fileWatcher = new ExternalFileWatcher();
            }
            fileWatcher.register(path);
        }
    }

    static class ExternalFileWatcher{

        private ExecutorService service = Executors.newFixedThreadPool(1, r -> {
            Thread t = new Thread(r);
            t.setDaemon(true);
            return t;
        });
        
        private final WatchService ws;
        private final AtomicBoolean stopped = new AtomicBoolean(true);
        private ConcurrentHashMap<WatchKey, Path> keys = new ConcurrentHashMap<>();

        public ExternalFileWatcher() throws IOException {
            ws = FileSystems.getDefault().newWatchService();
            service.submit(() -> {
                stopped.set(false);

                while (!stopped.get()) {
                    try {
                        WatchKey key = ws.take();
                        for (WatchEvent<?> watchEvent : key.pollEvents()) {
                            if (OVERFLOW == watchEvent.kind())
                                continue; // loop
                            
                            @SuppressWarnings("unchecked")
                            WatchEvent<Path> pathEvent = (WatchEvent<Path>) watchEvent;
                            Kind<Path> kind = pathEvent.kind();
                            Path parent = keys.get(key);
                            Path newPath = parent.resolve(pathEvent.context());
                            if (ENTRY_MODIFY == kind) {
                                LOGGER.info("New path modified: " + newPath); //$NON-NLS-1$
                                Resource resource = tempFiles.get(newPath);
                                if (resource != null) {
                                    GraphFileUtil.writeDataToGraph(newPath.toFile(), resource);
                                } else {
                                    LOGGER.warn("No resource found for {}", newPath.toAbsolutePath()); //$NON-NLS-1$
                                }
                            } else if (ENTRY_DELETE == kind) {
                                System.out.println("New path deleted: " + newPath); //$NON-NLS-1$
                            }
                        }
                        if (!key.reset()) {
                            keys.remove(key);
//                            break; // loop
                        }
                    } catch (InterruptedException e) {
                        if (!stopped.get())
                            LOGGER.error("Could not stop", e); //$NON-NLS-1$
                    } catch (Throwable t) {
                        LOGGER.error("An error occured", t); //$NON-NLS-1$
                    }
                }
            });
        }
        
        public void stop() {
            stopped.set(true);
        }
        
        public void register(Path path) throws IOException {
            if (Files.isDirectory(path)) {
                LOGGER.info("Registering path {}", path); //$NON-NLS-1$
                WatchKey key = path.toAbsolutePath().register(ws, ENTRY_DELETE, ENTRY_MODIFY);
                keys.put(key, path);
            }
        }
    }

    public static void stopFileWatcher() {
        synchronized (ExternalEditorAdapter.class) {
            tempFiles.clear();
            if (fileWatcher != null) {
                fileWatcher.stop();
                fileWatcher = null;
            }
        }
    }
}
