package org.simantics.fileimport.scl;

import java.io.IOException;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Optional;

import org.simantics.Simantics;
import org.simantics.databoard.util.Base64;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.service.SerialisationSupport;
import org.simantics.fileimport.Activator;
import org.simantics.fileimport.FileImportService;
import org.simantics.fileimport.dropins.FileImportDropins;
import org.simantics.layer0.Layer0;
import org.simantics.utils.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * SCL interface for Simantics File Import Functionality
 * See <code>"Dropins/Core"</code> SCL module for more
 * 
 * @author Jani Simomaa
 *
 */
public class DropinsSCL {
    
    private static final Logger LOGGER = LoggerFactory.getLogger(DropinsSCL.class);

    public static void watchDropinsFolder() {
        FileImportDropins.watchDropinsFolder();
    }

    public static void unwatchDropinsFolder() {
        FileImportDropins.unwatchDropinsFolder();
    }

    public static void uploadToDropinsBase64(String base64, String fileName) {
        // ensure that watcher is awake
        FileImportDropins.watchDropinsFolder();
        Path rootFolder = null;
        try {
            rootFolder = Activator.getDropinsFolder();
            Path newFile = rootFolder.resolve(fileName);
            if (Files.exists(newFile)) {
                newFile = findFreshFileName(rootFolder, fileName);
            }
            byte[] bytes = Base64.decode(base64);
            FileUtils.writeFile(newFile.toFile(), bytes);
        } catch (IOException e) {
            LOGGER.error("Could not upload base64 to file " + (rootFolder != null ? rootFolder.resolve(fileName).toAbsolutePath() : ""), e);
        }
    }

    private static Path findFreshFileName(Path root, String fileName) throws IOException {
        int ending = fileName.lastIndexOf('.');
        String glob = fileName;
        String suffix = "";
        if (ending > -1) {
            glob = fileName.substring(0, ending);
            suffix = fileName.substring(ending);
        }
        int i = 0;
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(root, glob)) {
            Iterator<Path> iter = stream.iterator();
            while (iter.hasNext()) {
                iter.next();
                i++;
            }
        }
        String newFileName = glob + "_" + i + suffix;
        return root.resolve(newFileName);
    }

    public static Map<String, Long> getUploadedFiles() throws DatabaseException {
        Map<String, String> results = FileImportService.getPathsAndResources();
        Map<String, Long> result = Simantics.getSession().syncRequest(new UniqueRead<Map<String, Long>>() {

            @Override
            public Map<String, Long> perform(ReadGraph graph) throws DatabaseException {
                Map<String, Long> map = new HashMap<>();
                for (Map.Entry<String, String> entry : results.entrySet()) {
                    String value = (String) entry.getValue();
                    Long id = Long.valueOf(value);
                    SerialisationSupport ss = graph.getService(SerialisationSupport.class);
                    try {
                        Resource r = ss.getResource(id);
                        String name = graph.getRelatedValue(r, Layer0.getInstance(graph).HasName);
                        map.put(name, id);
                    } catch (DatabaseException e) {
                        e.printStackTrace();
                    }
                }
                return map;
            }
        });
        return result;
    }

    public static void removeFileForId(long id) {
        FileImportService.removeFileForResource(id, Optional.empty());
    }

}
