/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db;

import org.simantics.db.exception.ServiceNotFoundException;

/**
 * <p>
 * A component with which one or more services are registered. The services can
 * be retrieved from this locator using the class representing the interface the
 * service must implement as a key. For example:
 * </p>
 * 
 * <pre>
 * SomeService service = session.getService(SomeService.class);
 * </pre>
 * 
 * <p>
 * If you want your service be disposed when the {@link ServiceLocator} is
 * disposed, make your service implement {@link Disposable}.
 * </p>
 * 
 * <p>
 * Implementations must be thread-safe.
 * </p>
 * 
 * <p>
 * This interface is not to be implemented or extended by clients.
 * </p>
 * 
 * <strong>NOTE:</strong> this is a blatant copy of IServiceLocator in
 * org.eclipse.ui.services, but for the purposes of the DB connection interface.
 * 
 * @see Disposable
 * 
 * @author eclipse.org
 * @author Tuukka Lehtonen
 */
public interface ServiceLocator {

    /**
     * Retrieves the service corresponding to the given API.
     * 
     * @param api This is the interface that the service implements and was
     *        registered with using {@link #registerService(Class, Object)}.
     *        Must not be <code>null</code>.
     * @return the requested service
     * @throws ServiceNotFoundException if a requested service is not available
     */
    public <T> T getService(Class<T> api) throws ServiceNotFoundException;

    /**
     * Tries to retrieve the service corresponding to the given API.
     * 
     * @param api
     *            This is the interface that the service implements. Must not be
     *            <code>null</code>.
     * @return The service, or <code>null</code> if no such service could be
     *         found.
     */
    public <T> T peekService(Class<T> api);

    /**
     * Whether this service exists within the scope of this service locator.
     * This does not include looking for the service within the scope of the
     * parents. This method can be used to determine whether a particular
     * service supports nesting in this scope.
     * 
     * @param api
     *            This is the interface that the service implements. Must not be
     *            <code>null</code>.
     * @return <code>true</code> iff the service locator can find a service
     *         for the given API; <code>false</code> otherwise.
     */
    public boolean hasService(Class<?> api);

    /**
     * @param api the api that must be implemented by the specified service
     * @param service the service implementation
     */
    public <T> void registerService(Class<T> api, T service);

}
