package fi.vtt.simantics.procore.internal;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.binding.Binding;
import org.simantics.db.IO;
import org.simantics.db.exception.RuntimeDatabaseException;
import org.simantics.db.impl.ClusterTraitsBase;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gnu.trove.set.hash.TLongHashSet;

public class ClusterSets {

    final static Logger LOGGER = LoggerFactory.getLogger(ClusterSets.class); 

	private IO write;
	final private String databaseId;
	final private HashMap<Long, Long> clusterSets; // Maps cluster set resource id to last user cluster id in the cluster set.
	final private HashMap<Long, Long> reverseMap;
	private int refCount; // Reference counter for user of this class.
	private boolean modified; // True if modified since last save.

	public void setWriteDirectory(IO write) {
		this.write = write;
	}

	public ClusterSets(IO read, IO write, String databaseId) {

		try {

			this.databaseId = databaseId;
			this.write = write;

			PersistentDataBlob blob = null;
			
			try {
				byte[] mainStateValue = read.readBytes(0, (int)read.length());
				ByteArrayInputStream bais = new ByteArrayInputStream(mainStateValue);
				blob = (PersistentDataBlob) org.simantics.databoard.Files.readFile(bais, PersistentDataBlob.BINDING);
				bais.close();
			} catch (IOException e1) {
				blob = new PersistentDataBlob();
				blob.clusterSets = new HashMap<>();
				blob.reverseMap = new HashMap<>();
				doSave(read, blob);
			}
			
			this.clusterSets = blob.clusterSets;
			this.reverseMap = blob.reverseMap;
			this.modified = false;
			
		} catch (Exception e) {
			
			throw new RuntimeDatabaseException("Failed to create ClusterSets.");
			
		}
	}
	public synchronized int inc() {
		return ++refCount;
	}
	public synchronized int dec() {
		return --refCount;
	}
	// Save is thread safe.
	public void dispose() {
		try {
			// We still save changes even if transaction is not finished.
			// This is because we have no cancel mechanism for cluster (sets).
			save();
		} catch (IOException e) {
			e.printStackTrace();
			throw new RuntimeDatabaseException("Failed to save ClusterSets.");
		} 
	}
	// clusterSets is not thread safe.
	public synchronized boolean containsKey(long resourceId) {
		return clusterSets.containsKey(resourceId);
	}
	public synchronized Long get(Long resourceId) {
		return clusterSets.get(resourceId);
	}
	public synchronized void put(long resourceId, long clusterId) {
		clusterSets.put(resourceId, clusterId);
		reverseMap.put(clusterId, resourceId);
		modified = true;
	}
	public synchronized Long getClusterSet(Long clusterId) {
		return reverseMap.get(clusterId);
	}
	public void clear() {
		
		for(Long key : clusterSets.keySet())
			clusterSets.put(key, -1L);
		
		touch();
		
	}
	
	public synchronized void touch() {
		modified = true;
	}
	
	private synchronized PersistentDataBlob doSave(IO io, PersistentDataBlob blob)  throws IOException {
		byte[] bytes = blob.bytes();
		write.saveBytes(bytes, bytes.length, true);
		return blob;
	}
	
	public synchronized void save() throws IOException {

		if (!modified)
			return;
		
		PersistentDataBlob blob = new PersistentDataBlob();
		blob.clusterSets = clusterSets;
		blob.reverseMap = reverseMap;
		doSave(write, blob);
		
		modified = false;

	}
	
	static public class PersistentDataBlob {
		final public static Binding BINDING = Bindings.getBindingUnchecked(PersistentDataBlob.class);
		public HashMap<Long, Long> clusterSets;
		public HashMap<Long, Long> reverseMap;
		public byte[] bytes() throws IOException {
			return Serialization.toByteArray(4096, BINDING, this);
		}
	}

	public void validate(long[] ids) {

		// clusterSets: cluster set persistent resource id (long) -> cluster id (long)
		// reverseMap: cluster id (long) -> cluster set persistent resource id (long)

		TLongHashSet idSet = new TLongHashSet(ids);

		ArrayList<Long> clusterSetResourceIds = new ArrayList<>(clusterSets.keySet());
		for(long setResourceId : clusterSetResourceIds) {
			long setClusterId = ClusterTraitsBase.getClusterIdFromResourceId(setResourceId);
			if(!idSet.contains(setClusterId)) {
				clusterSets.remove(setResourceId);
				LOGGER.info("Remove cluster set mapping: cluster set has been removed " + setResourceId);
			} else {
				long clusterId = clusterSets.get(setResourceId);
				if(clusterId == -1)
					continue;
				if(!idSet.contains(clusterId)) {
					clusterSets.put(setResourceId, -1L);
					LOGGER.info("Reset cluster set mapping: active cluster set cluster has been removed " + setResourceId + " " + clusterId);
				}
			}
		}
	
		ArrayList<Long> setClusterIds = new ArrayList<>(reverseMap.keySet());
		for(Long clusterId : setClusterIds) {
			if(clusterId == -1)
				continue;
			if(!idSet.contains(clusterId)) {
				reverseMap.remove(clusterId);
				LOGGER.info("Remove cluster set reverse mapping: cluster has been removed " + clusterId);
			} else {
				Long setResourceId = reverseMap.get(clusterId);
				long setClusterId = ClusterTraitsBase.getClusterIdFromResourceId(setResourceId);
				if(!idSet.contains(setClusterId)) {
					reverseMap.remove(clusterId);
					LOGGER.info("Remove cluster set reverse mapping: cluster set cluster has been removed " + setClusterId);
				}
			}
		}

	}

}
