/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.utils.ui.color;


import java.util.ArrayList;
import java.util.Collections;

import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;

/**
 * Widget to create and edit color gradients
 * 
 * @author Marko Luukkainen
 *
 */
public class ColorGradientComposite2 extends Composite implements ISelectionChangedListener{	
	
	private ColorGradientAdjustingCanvas gradientComposite;
	private Button addButton;
	private Button editButton;
	private Button removeButton;
	private Button rgbButton;
	private Button hsvButton;
	private int type = ColorGradient.RGB;
	
	private ArrayList<ColorValue> values = new ArrayList<ColorValue>();
	
	Composite typeComposite;
	Composite buttonComposite;
	
	public ColorGradientComposite2(Composite parent, int style) {
		super(parent,style);
		GridLayout layout = new GridLayout(2,false);
	    this.setLayout(layout);
	    gradientComposite = new ColorGradientAdjustingCanvas(this,SWT.HORIZONTAL);
	    
	    gradientComposite.addSelectionChangedListener(this);
	    
	    typeComposite = new Composite(this, SWT.NONE);
	    
	    typeComposite.setLayout(new GridLayout(1,false));
	    rgbButton = new Button(typeComposite,SWT.RADIO);
	    rgbButton.setSelection(true);
	    rgbButton.addSelectionListener(new SelectionListener() {
	    	public void widgetDefaultSelected(SelectionEvent e) {
	    		widgetSelected(e);
	    	}
	    	public void widgetSelected(SelectionEvent e) {
	    		rgbButton.setSelection(true);
	    		hsvButton.setSelection(false);
	    		type = ColorGradient.RGB;
	    		gradientComposite.setGradient(new ColorGradient(values,type));
	    	}
	    });
	    rgbButton.setText("RGB");
	    hsvButton = new Button(typeComposite,SWT.RADIO);
	    hsvButton.addSelectionListener(new SelectionListener() {
	    	public void widgetDefaultSelected(SelectionEvent e) {
	    		widgetSelected(e);
	    	}
	    	public void widgetSelected(SelectionEvent e) {
	    		hsvButton.setSelection(true);
	    		rgbButton.setSelection(false);
	    		type = ColorGradient.HSV;
	    		gradientComposite.setGradient(new ColorGradient(values,type));
	    	}
	    });
	    hsvButton.setText("HSV"); 
	    
	    buttonComposite = new Composite(this,SWT.NONE);

	    buttonComposite.setLayout(new FillLayout(SWT.HORIZONTAL));
	    
	    addButton = new Button(buttonComposite,SWT.PUSH);
	    addButton.setText("Add new color");
	    addButton.addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() { 
        	public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {    
        		RGB rgb = openDialog(null);
                if (rgb != null) {
                    Color c = new Color(rgb.red,rgb.green,rgb.blue);
                    Double value;
                    if (values.size() == 0) {
                    	value = 0.0;
                    } else if (values.size() == 1) {
                    	value = 100.0;
                    } else {
                    	StructuredSelection selection = (StructuredSelection)gradientComposite.getSelection();
                    	if (selection.size() == 1) {
                    		// add new color next to the selection
                    		ColorValue v = (ColorValue)selection.getFirstElement();
                    		int index = values.indexOf(v);
                    		if (index == values.size() -1) {
                    			index--;
                    		}
                    		value = (values.get(index+1).getValue()-values.get(index).getValue())*0.5;
                    		value += values.get(index).getValue();
                    	} else {
                    		// add new color to largest gap
	                    	int index = 0;
	                    	double r = 0.0;
	                    	for (int i = 0; i < values.size() -1; i++) {
	                    		double v1 = values.get(i).getValue();
	                    		double v2 = values.get(i+1).getValue();
	                    		double vr = v2 -v1;
	                    		if (vr > r) {
	                    			r=vr;
	                    			index = i;
	                    		}
	                    	}
	                    	value = values.get(index).getValue() + r *0.5;
                    	}
                    }
                    addColor(c,value);
                }
            }
        });
	    editButton = new Button(buttonComposite,SWT.PUSH);
	    editButton.setText("Edit color");
	    editButton.addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() { 
        	public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {    
        	
    			Object obj = ((StructuredSelection)gradientComposite.getSelection()).getFirstElement();
    			if (obj == null)
        			return;
    			if (obj instanceof ColorValue) {
    				RGB rgb = openDialog(((ColorValue)obj).getColor().getRgb());
                    if (rgb != null) {
                    	modifyColorValueColor((ColorValue)obj,rgb);
                    }
    			}    			
        		
        	}
        });
	    editButton.setEnabled(false);
	    removeButton = new Button(buttonComposite,SWT.PUSH);
	    removeButton.setText("Remove color");
	    removeButton.addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() { 
        	public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {    
        		Object o = ((StructuredSelection)gradientComposite.getSelection()).getFirstElement();
        		if (o == null)
        			return;
    			if (o instanceof ColorValue) {
    				values.remove(o);
    				updateWidgets();
    			}    			
        		
        	}
        });
	    removeButton.setEnabled(false);
	    
	    GridDataFactory.fillDefaults().span(1, 1).grab(true, false).align(SWT.FILL, SWT.CENTER).hint(SWT.DEFAULT, 32).applyTo(gradientComposite);
	    GridDataFactory.fillDefaults().grab(false, true).align(SWT.LEFT, SWT.FILL).applyTo(typeComposite);
	    GridDataFactory.fillDefaults().span(1, 1).grab(true, false).align(SWT.FILL, SWT.TOP).applyTo(buttonComposite);  
	}
	
	protected RGB openDialog(RGB selection) {
//		ColorDialog dialog = new ColorDialog(ColorGradientComposite2.this.getShell(), SWT.NONE);
//        if (selection != null)
//        	dialog.setRGB(selection);
//        RGB rgb = dialog.open();
//        return rgb;
		org.simantics.utils.ui.color.ColorDialog dialog = new org.simantics.utils.ui.color.ColorDialog(getShell());
		if (selection != null) {
			Color color = new Color(selection);
			dialog.setInitialValue(color);
		}
		if (dialog.open() == org.simantics.utils.ui.color.ColorDialog.OK) {
			Color color = dialog.getColor();
			return color.getRgb();
		}
		return null;
	}

	public void addColor(Color color, double value) {
		addColor(new ColorValue(color,value));
	}
	
	public void addColor(ColorValue value) {
		values.add(value);
		updateWidgets();
	}
	
	private void updateWidgets() {
		Collections.sort(values,new ColorValueComparator());	
		gradientComposite.setGradient(new ColorGradient(values,type));
	}

	public ColorGradient getGradient() {
		return new ColorGradient(values,type);
	}

	public void dispose() {
		super.dispose();
	}
	
	public void setGradient(ColorGradient gradient) {
		values.clear();
		type = gradient.getType();
		for (ColorValue value : gradient.getColorValues())
			addColor(value);
		if (type == ColorGradient.HSV) {
			rgbButton.setSelection(false);
			hsvButton.setSelection(true);
		} else if (type == ColorGradient.RGB) {
			hsvButton.setSelection(false);
			rgbButton.setSelection(true);
		}
		
	}
	
	private void modifyColorValueColor(ColorValue cValue, RGB rgb) {
		values.remove(cValue);;
		Color newColor = new Color(rgb.red,rgb.green,rgb.blue);
		ColorValue newCValue = new ColorValue(newColor,cValue.getValue());
		values.add(newCValue);
		updateWidgets();
	}
	
	/**
	 * Enables and disables "Edit color" and "Remove color" buttons depending on selected item
	 */
	public void selectionChanged(SelectionChangedEvent event) {
		Object obj = ((IStructuredSelection)event.getSelection()).getFirstElement();
		if (obj == null) {
			editButton.setEnabled(false);
			removeButton.setEnabled(false);
		} else {
			editButton.setEnabled(true);
			int index = values.indexOf(obj);
			if (index > 0 && index < values.size() -1)
				removeButton.setEnabled(true);
			else
				removeButton.setEnabled(false);
		}
	}
	
	public Composite getTypeComposite() {
		return typeComposite;
	}
	
	public Composite getButtonComposite() {
		return buttonComposite;
	}
	
}

	
