/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.history.test;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;

import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Order;
import org.junit.jupiter.api.Test;
import org.simantics.databoard.Bindings;
import org.simantics.databoard.Datatypes;
import org.simantics.databoard.accessor.StreamAccessor;
import org.simantics.databoard.accessor.error.AccessorException;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.impl.DoubleArrayBinding;
import org.simantics.databoard.type.ArrayType;
import org.simantics.databoard.type.RecordType;
import org.simantics.databoard.util.Bean;
import org.simantics.databoard.util.Limit;
import org.simantics.databoard.util.Range;
import org.simantics.history.Collector;
import org.simantics.history.History;
import org.simantics.history.HistoryException;
import org.simantics.history.HistoryManager;
import org.simantics.history.csv.CSVFormatter;
import org.simantics.history.impl.CollectorImpl;
import org.simantics.history.impl.CollectorState;
import org.simantics.history.util.ProgressMonitor;
import org.simantics.history.util.Stream;
import org.simantics.history.util.StreamIterator;
import org.simantics.history.util.ValueBand;
import org.simantics.history.util.subscription.SamplingFormat;
import org.simantics.history.util.subscription.SubscriptionItem;
import org.simantics.utils.FileUtils;

public class TestHistory {
	
	static final double NaN = Double.NaN;
	
	// Time            0.0  0.1  0.2  0.3  0.4  0.5  0.6  0.7  0.8  0.9  1.0  1.1  1.2  1.3  1.4
	double[] data1 = { 5.0, 5.0, 5.0, 5.0, 6.0, 6.0, 6.0, 7.0, 8.0, 9.0, NaN,10.0, NaN, NaN, NaN };	
	double[] data2 = { 5.0, 5.0, 5.0, 5.0, 6.0, 6.0, 6.0, 7.0, 8.0, 9.0,10.0, 9.0, 5.0, 4.0, 3.0 };
	byte[] data3 = { 5, 6, 7, 8, 9, 10, 9, 5, 4, 3 };
	
	// Subscription formats
	SamplingFormat simple, allfields, vector, minmax, byteformat, string;

	// History
	HistoryManager historian;
	File workarea;
	Collector collector;

	@BeforeEach
	@Order(1)
	public void initSubscriptionFormats()
	{
		simple = new SamplingFormat();
		simple.formatId = "simple";
		RecordType format;
		format = (RecordType) (simple.format = new RecordType());
		format.addComponent("time", Datatypes.FLOAT);
		format.addComponent("endTime", Datatypes.FLOAT);
		format.addComponent("value", Datatypes.DOUBLE );
		format.addComponent("quality", Datatypes.BYTE );
		simple.interval = NaN;
		simple.deadband = NaN;

		string = new SamplingFormat();
		string.formatId = "string";
		format = (RecordType) (string.format = new RecordType());
		format.addComponent("time", Datatypes.FLOAT);
		format.addComponent("endTime", Datatypes.FLOAT);
		format.addComponent("value", Datatypes.STRING );
		format.addComponent("quality", Datatypes.BYTE );
		string.interval = NaN;
		string.deadband = NaN;
		
		allfields = new SamplingFormat();
		allfields.formatId = "alldata"; 
		format = (RecordType) (allfields.format = new RecordType());
		format.addComponent("time", Datatypes.DOUBLE);
		format.addComponent("endTime", Datatypes.DOUBLE);
		format.addComponent("value", Datatypes.DOUBLE );
		format.addComponent("lastValue", Datatypes.DOUBLE);		
		format.addComponent("min", Datatypes.DOUBLE);
		format.addComponent("max", Datatypes.DOUBLE);
		format.addComponent("avg", Datatypes.DOUBLE);
		format.addComponent("median", Datatypes.DOUBLE);
		format.addComponent("quality", Datatypes.BYTE );
		format.addComponent("count", Datatypes.INTEGER);
		allfields.interval = NaN;
		allfields.deadband = NaN;		

		byteformat = new SamplingFormat();
		byteformat.formatId = "byte";
		byteformat.format = new RecordType();
		format = (RecordType) (byteformat.format = new RecordType());
		format.addComponent("time", Datatypes.DOUBLE);
		format.addComponent("endTime", Datatypes.DOUBLE);
		format.addComponent("value", Datatypes.BYTE);
		format.addComponent("lastValue", Datatypes.BYTE);		
		format.addComponent("min", Datatypes.BYTE);
		format.addComponent("max", Datatypes.BYTE);
		format.addComponent("avg", Datatypes.DOUBLE);
		format.addComponent("median", Datatypes.BYTE);
		format.addComponent("quality", Datatypes.BYTE);
		format.addComponent("count", Datatypes.INTEGER);
		byteformat.interval = NaN;
		byteformat.deadband = NaN;		
		
		vector = new SamplingFormat();
		vector.formatId = "vector";
		vector.format = new RecordType();
		format = (RecordType) (vector.format = new RecordType());
		format.addComponent("time", Datatypes.FLOAT);
		format.addComponent("endTime", Datatypes.FLOAT);
		format.addComponent("value", new ArrayType( Datatypes.DOUBLE, Range.between(Limit.inclusive(3), Limit.inclusive(3)) ));
		format.addComponent("count", Datatypes.INTEGER);
		vector.interval = NaN;
		vector.deadband = NaN;
		
		minmax = new SamplingFormat();
		minmax.formatId = "minmax";
		minmax.format = new RecordType();
		format = (RecordType) (minmax.format = new RecordType());
		format.addComponent("time", Datatypes.FLOAT);
		format.addComponent("endTime", Datatypes.FLOAT);
		format.addComponent("value", Datatypes.DOUBLE );
		format.addComponent("min", Datatypes.DOUBLE );
		format.addComponent("max", Datatypes.DOUBLE );
		minmax.interval = Double.MAX_VALUE;
		minmax.deadband = Double.MAX_VALUE;		
	}

	@BeforeEach
	@Order(2)
	public void initHistory() {
		CollectorState cs = new CollectorState();
		workarea = FileUtils.createTmpDir();
		System.out.println(workarea);
		historian = History.openFileHistory( workarea );
//		historian = History.createMemoryHistory();
		collector = new CollectorImpl( historian );
	}

	@AfterEach
	public void uninitHistory() {
		if ( collector != null ) {
			collector.close();			
		}
		if ( historian != null ) {
			historian.close();
		}
		if ( workarea != null) {
			try {
				FileUtils.deleteAll(workarea);
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
	}
	
	@Test
	public void testGetModifySubscription() throws Exception
	{
		// 1. Create Item and Collector
		SubscriptionItem hi = SubscriptionItem.createItem("NotMyVariable", "MySubscription", allfields);
		historian.create(hi);
		collector.addItem(hi);
		
		// 2. Get Item
		SubscriptionItem[] items = collector.getItems();		
		Assertions.assertEquals(1, items.length);
		SubscriptionItem hii = new SubscriptionItem();
		hii.readAvailableFields( items[0] );
		Assertions.assertTrue( hii.equalContents( hi ) );
		
		// 3. Modify Item
		hi.variableId = "MyVariable";
		collector.setItem(hi);
		items = collector.getItems();		
		Assertions.assertEquals(1, items.length);
		hii.readAvailableFields( items[0] );
		Assertions.assertTrue( hii.equalContents( hi ) );
		
		// 4. Remove & Add
		collector.removeItem(hi.id);
		Assertions.assertEquals(0, collector.getItems().length);
		collector.addItem(hi);
		
		// 5. Open & Modify subscription
		collector.beginStep(Bindings.DOUBLE, 5.0);
		hi.interval = 1.2;
		collector.setItem(hi);
		collector.endStep();
		collector.close();

		// 6. Read from history
		Bean bean = historian.getItem(hi.id);
		hi.interval = 0.0;
		hi.readAvailableFields(bean);
		Assertions.assertEquals(1.2, hi.interval, 0.1);
	}

	@Test
	public void failtestRecreateSubscription() throws HistoryException
	{
		Collector collector2;
		SubscriptionItem hi = SubscriptionItem.createItem("MyVariable", "MySubscription", allfields);
		historian.create(hi);
		
		collector.addItem( hi );
		collector2 = new CollectorImpl( historian );
		collector2.addItem( hi );
		
		try {
			collector.beginStep(Bindings.DOUBLE, 0.0);
			collector2.beginStep(Bindings.DOUBLE, 0.0);
			Assertions.fail("Recreate subscription should have failed");
		} catch (HistoryException e) {
			// expected exception
		}
	
		collector2.close();
	}

	@Test
	public void failtestReopenStream() throws HistoryException
	{
		// 1. Create history and collector item
		SamplingFormat format = allfields.clone();
		format.deadband = 2.5;
		SubscriptionItem[] his = SubscriptionItem.createItems("MyVariable", "MySubscription", allfields, minmax);
		historian.create(his);
		StreamAccessor sa1=null, sa2=null;
		try {
			sa1 = historian.openStream(his[0].id, "r");
			sa2 = historian.openStream(his[1].id, "r");
			Assertions.fail("HistoryManager must not allow to open two handles to same subscription");
		} catch (HistoryException he) {
			// Expected exception
		}
		try {
		if (sa1!=null)sa1.close();
		if (sa2!=null) sa2.close();
		} catch (AccessorException e) {
			throw new HistoryException(e);
		}
		
		historian.delete( his[0].id, his[1].id );
	}
	
	@Test
	public void testDisableItem() throws Exception {
		// 1. Create history and collector item
		SamplingFormat format = allfields.clone();
		format.deadband = 2.5;
		SubscriptionItem[] hi = SubscriptionItem.createItems("MyVariable", "MySubscription", allfields, minmax);
		historian.create(hi);		
		
		// 2. Create collector
		collector.addItems(hi);
		
		// Write data
		try {
			// Simulate
			double[] data = data1;
			for (int i=0; i<4; i++) 
			{
				collector.beginStep(Bindings.DOUBLE, i * 0.1);
				collector.setValue("MyVariable", Bindings.DOUBLE, data[i]);
				collector.endStep();
			}
			
			// Disable item
			hi[0].enabled = false;
			collector.setItem(hi[0]);
			for (int i=4; i<12; i++) 
			{
				collector.beginStep(Bindings.DOUBLE, i * 0.1);
				collector.setValue("MyVariable", Bindings.DOUBLE, data[i]);
				collector.endStep();
			}
			
			// Enable item
			hi[0].enabled = true;
			collector.setItem(hi[0]);
			for (int i=12; i<data.length; i++) 
			{
				collector.beginStep(Bindings.DOUBLE, i * 0.1);
				collector.setValue("MyVariable", Bindings.DOUBLE, data[i]);
				collector.endStep();
			}			
			
		} finally {
			collector.close();
		}
		
		
		// 3. Verify data
		StreamAccessor aa = historian.openStream(hi[0].id, "r");
		try {
			Binding sampleBinding = Bindings.getBinding( allfields.format );
			
			// Read and assert the sample
			Object sample = aa.get(0, sampleBinding);
			ValueBand se = new ValueBand( sampleBinding, sample );
			if (se.supportsNullValue()) { 
				Assertions.assertEquals(0.0, (Double) se.getTime( Bindings.DOUBLE ), 0.01);
				Assertions.assertEquals(0.3, (Double) se.getEndTime( Bindings.DOUBLE ), 0.01);
				Assertions.assertFalse( se.isNullValue() );
				
				aa.get(1, sampleBinding, sample);
				Assertions.assertEquals(0.4, (Double) se.getTime( Bindings.DOUBLE ), 0.01);
				Assertions.assertEquals(1.1, (Double) se.getEndTime( Bindings.DOUBLE ), 0.01);
				Assertions.assertTrue( se.isNullValue() );
	
				aa.get(2, sampleBinding, sample);			
				Assertions.assertEquals(1.2, (Double) se.getTime( Bindings.DOUBLE ), 0.01);
			}
			
		} finally {
			aa.close();
		}

		// 4. Verify min-max has only 1 sample
		aa = historian.openStream(hi[1].id, "r");
		try {
			Assertions.assertEquals(1, aa.size());
			Binding sampleBinding = Bindings.getBinding( minmax.format );
			
			// Read and assert the sample
			Object sample = aa.get(0, sampleBinding);
			ValueBand se = new ValueBand( sampleBinding, sample );
			Assertions.assertEquals(0.0, (Double) se.getTime( Bindings.DOUBLE ), 0.01);
			Assertions.assertEquals(1.4, (Double) se.getEndTime( Bindings.DOUBLE ), 0.01);
			Assertions.assertEquals(5.0, (Double) se.getMin( Bindings.DOUBLE ), 0.01);
			Assertions.assertEquals(10.0, (Double) se.getMax( Bindings.DOUBLE ), 0.01);
			
		} finally {
			aa.close();
		}
	}

	
	@Test
	public void testDeadband() throws Exception {
		// 1. Create history and collector item
		SamplingFormat format = allfields.clone();
		format.deadband = 2.5;
		SubscriptionItem hi = SubscriptionItem.createItem("MyVariable", "MySubscription", format);
		historian.create(hi);		
		
		// 2. Create collector
		collector.addItem(hi);		
		try {
			// Add values, dt=0.1
			double[] data = data2;
			for (int i=0; i<data.length; i++) 
			{
				collector.beginStep(Bindings.DOUBLE, i * 0.1);
				collector.setValue("MyVariable", Bindings.DOUBLE, data[i]);
				collector.endStep();
			}			
		} finally {
			collector.close();
		}

		// 3. Verify data
		StreamAccessor aa = historian.openStream(hi.id, "r");		
		try {
			Binding sampleBinding = Bindings.getBinding( format.format );
			// Read and assert the sample
			Object sample = aa.get(0, sampleBinding);
			ValueBand se = new ValueBand( sampleBinding, sample );
			Assertions.assertEquals(0.0, (Double) se.getTime( Bindings.DOUBLE ), 0.01);
			Assertions.assertEquals(0.7, (Double) se.getEndTime( Bindings.DOUBLE ), 0.01);
			Assertions.assertEquals(5.0, (Double) se.getValue( Bindings.DOUBLE ), 0.01);
			Assertions.assertEquals(5.0, (Double) se.getMedian( Bindings.DOUBLE ), 0.01);
			Assertions.assertEquals(5.0, (Double) se.getMin( Bindings.DOUBLE ), 0.01);
			Assertions.assertEquals(7.0, (Double) se.getMax( Bindings.DOUBLE ), 0.01);
			Assertions.assertEquals(8, se.getCount());
			
			aa.get(1, sampleBinding, sample);
			Assertions.assertEquals(0.8, (Double) se.getTime( Bindings.DOUBLE ), 0.01);
			Assertions.assertEquals(1.1, (Double) se.getEndTime( Bindings.DOUBLE ), 0.01);
			Assertions.assertEquals(8.0, (Double) se.getValue( Bindings.DOUBLE ), 0.01);
			Assertions.assertEquals(9.0, (Double) se.getMedian( Bindings.DOUBLE ), 0.01);
			Assertions.assertEquals(8.0, (Double) se.getMin( Bindings.DOUBLE ), 0.01);
			Assertions.assertEquals(10.0, (Double) se.getMax( Bindings.DOUBLE ), 0.01);
			Assertions.assertEquals(4, se.getCount());

			aa.get(2, sampleBinding, sample);
			Assertions.assertEquals(1.2, (Double) se.getTime( Bindings.DOUBLE ), 0.01);
			Assertions.assertEquals(1.4, (Double) se.getEndTime( Bindings.DOUBLE ), 0.01);			
			Assertions.assertEquals(5.0, (Double) se.getValue( Bindings.DOUBLE ), 0.01);
			Assertions.assertEquals(4.0, (Double) se.getMedian( Bindings.DOUBLE ), 0.01);
			Assertions.assertEquals(3.0, (Double) se.getMin( Bindings.DOUBLE ), 0.01);
			Assertions.assertEquals(5.0, (Double) se.getMax( Bindings.DOUBLE ), 0.01);
			Assertions.assertEquals(3, se.getCount());
		} finally {
			aa.close();
		}
	}

	
	@Test
	public void testMedianAndInterval() throws Exception {

		// 1. Create history and collector item
		SamplingFormat format = allfields.clone();
		format.interval = 1.0;
		SubscriptionItem hi = SubscriptionItem.createItem("MyVariable", "MySubscription", format);
		historian.create(hi);
		// Create binding for the sample format
		Binding sampleBinding = Bindings.getBinding( format.format );
		
		
		// 2. Create collector
		collector.addItem( hi );

		// Write data
		try {
			// Add values, dt=0.1
			double[] data = data2;
			for (int i=0; i<data.length; i++) 
			{
				collector.beginStep(Bindings.DOUBLE, i * 0.1);
				collector.setValue("MyVariable", Bindings.DOUBLE, data[i]);
				collector.endStep();
			}			
		} finally {
			collector.close();
		}
		
		// Verify data
		StreamAccessor aa = historian.openStream(hi.id, "r");		
		try {
			// 10s interval. There should be only one entry
			Assertions.assertEquals(2, aa.size());
			
			// Read and assert the sample
			Object sample = aa.get(0, sampleBinding);
			ValueBand se = new ValueBand( sampleBinding, sample );
			Assertions.assertEquals(0.0, (Double) se.getTime( Bindings.DOUBLE ), 0.01);
			Assertions.assertEquals(0.9, (Double) se.getEndTime( Bindings.DOUBLE ), 0.01);
			Assertions.assertEquals(6.0, (Double) se.getMedian( Bindings.DOUBLE ), 0.01);
			
			aa.get(1, sampleBinding, sample);
			Assertions.assertEquals(1.0, (Double) se.getTime( Bindings.DOUBLE ), 0.01);
			Assertions.assertEquals(1.4, (Double) se.getEndTime( Bindings.DOUBLE ), 0.01);
			Assertions.assertEquals(5.0, (Double) se.getMedian( Bindings.DOUBLE ), 0.01);
			
		} finally {
			aa.close();
		}
		
		// Reopen subscription
		// Write data
		collector = new CollectorImpl( historian );				
		try {
			// Add values, dt=0.1
			double[] data = data2;
			for (int i=0; i<data.length; i++) 
			{
				collector.beginStep(Bindings.DOUBLE, (i+data.length) * 0.1);
				collector.setValue("MyVariable", Bindings.DOUBLE, data[i]);
				collector.endStep();
			}			
		} finally {
			collector.close();
		}
		
		// Verify data
		aa = historian.openStream(hi.id, "r");		
		try {
			Assertions.assertEquals(3, aa.size());
			
			// Read and assert the sample
			Object sample = aa.get(0, sampleBinding);
			ValueBand se = new ValueBand( sampleBinding, sample );
			Assertions.assertEquals(0.0, (Double) se.getTime( Bindings.DOUBLE ), 0.01);
			Assertions.assertEquals(0.9, (Double) se.getEndTime( Bindings.DOUBLE ), 0.01);
			Assertions.assertEquals(6.0, (Double) se.getMedian( Bindings.DOUBLE ), 0.01);
			
			aa.get(1, sampleBinding, sample);
			Assertions.assertEquals(1.0, (Double) se.getTime( Bindings.DOUBLE ), 0.01);
			Assertions.assertEquals(2.1, (Double) se.getEndTime( Bindings.DOUBLE ), 0.01);
			Assertions.assertEquals(5.0, (Double) se.getMedian( Bindings.DOUBLE ), 0.01);
			
			aa.get(2, sampleBinding, sample);
			Assertions.assertEquals(2.2, (Double) se.getTime( Bindings.DOUBLE ), 0.01);
			Assertions.assertEquals(2.9, (Double) se.getEndTime( Bindings.DOUBLE ), 0.01);
			Assertions.assertEquals(7.0, (Double) se.getMedian( Bindings.DOUBLE ), 0.01);
			
		} finally {
			aa.close();
		}
				
		historian.delete(hi.id);
	}

	@Test
	public void testAccessor() throws Exception
	{
		// 1. Create history and collector item
		SamplingFormat format = allfields.clone();
		SubscriptionItem hi = SubscriptionItem.createItem("MyVariable", "MySubscription", format);
		historian.create(hi);
		
		// 2. Create collector
		collector.addItem(hi);
		StreamAccessor aa = historian.openStream(hi.id, "r");		
		try {
			// Write data
			double[] data = data2;
			for (int i=0; i<data.length; i++) 
			{
				collector.beginStep(Bindings.DOUBLE, i * 0.1);
				collector.setValue("MyVariable", Bindings.DOUBLE, data[i]);
				collector.endStep();
			}			

			// Assert data
			aa.reset();
			Assertions.assertEquals(10, aa.size());
			
			// Write some more
			for (int i=0; i<data.length; i++) 
			{
				collector.beginStep(Bindings.DOUBLE, (i + data.length) * 0.1);
				collector.setValue("MyVariable", Bindings.DOUBLE, data[i]);
				collector.endStep();
			}			
			
			// Assert data
			aa.reset();
			Assertions.assertEquals(20, aa.size());

			// Print
			Binding sampleBinding = Bindings.getBinding( format.format ); 			
			for (int i=0; i<aa.size(); i++) {
				System.out.println( i+": "+sampleBinding.toString( aa.get(i, sampleBinding) ) );
			}
			
			
		} finally {
			aa.close();
			collector.close();
		}
		
	}

	@Test
	public void testArrayData() throws Exception
	{
		// 1. Create history and collector item
		SamplingFormat format = vector.clone();
		SubscriptionItem hi = SubscriptionItem.createItem("MyVariable", "MySubscription", format);
		historian.create(hi);
		
		// 2. Create collector
		collector.addItem( hi );
		ArrayType vectorType = (ArrayType) ((RecordType)format.format).getComponent("value").type;
		Binding vectorBinding = DoubleArrayBinding.createFrom( vectorType );
		
		// Write data
		int count = 20;
		try {
			// Simulate
			double[] vector = new double[] { 1.0, 2.0, 3.0 };
			
			for (int i=0; i<count; i++) 
			{
				collector.beginStep(Bindings.DOUBLE, i * 0.1);
				
				vector[0] = i;
				vector[1] = i*1.5+1;
				vector[2] = i*-2.4+5;
				
				collector.setValue("MyVariable", vectorBinding, vector);
				collector.endStep();
			}
			
		} finally {
			collector.close();
		}
		
		StreamAccessor aa = historian.openStream(hi.id, "r");
		try {
			Assertions.assertEquals(count, aa.size());
		} finally {
			aa.close();
		}
		
		historian.delete(hi.id);
	}

	@Test
	public void testStringData() throws Exception
	{
		// 1. Create history and collector item
		SamplingFormat format = string.clone();
		SubscriptionItem hi = SubscriptionItem.createItem("MyVariable", "MySubscription", format);
		historian.create(hi);
		
		// 2. Create collector
		collector.addItem( hi );
		Binding lineBinding = Bindings.STRING;
		
		// Write data
		int count = 20;
		try {
			// Simulate
			for (int i=0; i<count; i++) 
			{
				collector.beginStep(Bindings.DOUBLE, i * 0.1);
				collector.setValue("MyVariable", lineBinding, "Line: "+i);
				collector.endStep();
			}
			
		} finally {
			collector.close();
		}
		
		StreamAccessor aa = historian.openStream(hi.id, "rw");
		try {
			// Assert data ok
			Binding beanBinding = Bindings.getBeanBinding(hi.format);
			Assertions.assertEquals(count, aa.size());
			for (int i=0; i<count; i++) {
				Bean bean = (Bean) aa.get(i, beanBinding);
				String line = (String) bean.getField("value");
				Assertions.assertEquals("Line: "+i, line);
			}
			
			// Delete entries
			aa.remove(10, 2);
			Assertions.assertEquals(count-2, aa.size());
			for (int i=0; i<10; i++) {
				Bean bean = (Bean) aa.get(i, beanBinding);
				String line = (String) bean.getField("value");
				Assertions.assertEquals("Line: "+i, line);
			}
			for (int i=10; i<18; i++) {
				Bean bean = (Bean) aa.get(i, beanBinding);
				String line = (String) bean.getField("value");
				Assertions.assertEquals("Line: "+(i+2), line);
			}
			
		} finally {
			aa.close();
		}
		
		historian.delete(hi.id);
	}
	

	@Test
	public void testMinMax() throws Exception {
		// 1. Create history and collector item
		SamplingFormat format = minmax.clone();
		SubscriptionItem hi = SubscriptionItem.createItem("MyVariable", "MySubscription", format);
		historian.create(hi);
		
		// 2. Create collector
		collector.addItem( hi );

		// 3. Write data
		try {
			// Add values, dt=0.1
			double[] data = data1;
			for (int i=0; i<data.length; i++) 
			{
				collector.beginStep(Bindings.DOUBLE, i * 0.1);
				collector.setValue("MyVariable", Bindings.DOUBLE, data[i]);
				collector.endStep();
			}			
		} finally {
			collector.close();
		}

		// 3. Verify data
		StreamAccessor aa = historian.openStream(hi.id, "r");
		try {
			Assertions.assertEquals(1, aa.size());
			Binding sampleBinding = Bindings.getBinding( format.format );
			// Read and assert the sample
			Object sample = aa.get(0, sampleBinding);
			ValueBand se = new ValueBand( sampleBinding, sample );
			Assertions.assertEquals(0.0, (Double) se.getTime( Bindings.DOUBLE ), 0.01);
			Assertions.assertEquals(1.4, (Double) se.getEndTime( Bindings.DOUBLE ), 0.01);
			Assertions.assertEquals(5.0, (Double) se.getMin( Bindings.DOUBLE ), 0.01);
			Assertions.assertEquals(10.0, (Double) se.getMax( Bindings.DOUBLE ), 0.01);
		} finally {
			aa.close();
		}
		
	}
	
	
	public @Test void testStream() throws Exception {
		// 1. Create history and collector item
		SubscriptionItem hi = SubscriptionItem.createItem("MyVariable", "MySubscription", SamplingFormat.allfields);
		historian.create(hi);
		
		// 2. Create collector
		collector.addItem( hi );

		// 3. Write and assert data
		StreamAccessor aa = historian.openStream(hi.id, "r");
		try {
			// Write data
			double[] data = data2;
			for (int i=0; i<data.length; i++) 
			{
				collector.beginStep(Bindings.DOUBLE, i * 0.1);
				collector.setValue("MyVariable", Bindings.DOUBLE, data[i]);
				collector.endStep();
			}			
			
			// Assert data
			aa.reset();
			Stream stream = new Stream(aa);
			
			Object sample0 = aa.get(0, stream.sampleBinding);
			Object sample1 = aa.get(1, stream.sampleBinding);
			Object sample;
			
			sample = stream.getLowerSample(Bindings.DOUBLE, 0.0);
			Assertions.assertNull(sample);
			
			sample = stream.getLowerSample(Bindings.DOUBLE, 0.05);
			Assertions.assertTrue( stream.sampleBinding.equals(sample, sample0) );

			sample = stream.getFloorSample(Bindings.DOUBLE, 0.05);
			Assertions.assertTrue( stream.sampleBinding.equals(sample, sample0) );

			sample = stream.getFloorSample(Bindings.DOUBLE, 0.00);
			Assertions.assertTrue( stream.sampleBinding.equals(sample, sample0) );
			
			sample = stream.getSample(Bindings.DOUBLE, 0.05);
			Assertions.assertNull(sample);

			sample = stream.getSample(Bindings.DOUBLE, 0.00);
			Assertions.assertTrue( stream.sampleBinding.equals(sample, sample0) );

			sample = stream.getCeilingSample(Bindings.DOUBLE, -0.10);
			Assertions.assertTrue( stream.sampleBinding.equals(sample, sample0) );
			
			sample = stream.getCeilingSample(Bindings.DOUBLE, 0.00);
			Assertions.assertTrue( stream.sampleBinding.equals(sample, sample0) );

			sample = stream.getCeilingSample(Bindings.DOUBLE, 0.05);
			Assertions.assertTrue( stream.sampleBinding.equals(sample, sample1) );

			sample = stream.getHigherSample(Bindings.DOUBLE, -0.05);
			Assertions.assertTrue( stream.sampleBinding.equals(sample, sample0) );

			sample = stream.getHigherSample(Bindings.DOUBLE, 0.00);
			Assertions.assertTrue( stream.sampleBinding.equals(sample, sample1) );
			
			sample = stream.getHigherSample(Bindings.DOUBLE, 0.05);
			Assertions.assertTrue( stream.sampleBinding.equals(sample, sample1) );
			
			sample = stream.getHigherSample(Bindings.DOUBLE, 1.45);
			Assertions.assertNull( sample );
		} finally {
			aa.close();
			collector.close();
		}
	}
	

	@Test
	public void testByteData() throws Exception {
		// 1. Create stream
		SamplingFormat format = byteformat.clone();
		format.formatId = "test";
		SubscriptionItem hi = SubscriptionItem.createItem("MyVariable", "MySubscription", format);
		historian.create( hi );
		
		// 2. Create collector
		collector.addItem(hi);

		// 3. Write and assert data
		try {
			// Write data
			double[] data = data2;
			for (int i=0; i<data.length; i++) 
			{
				collector.beginStep(Bindings.DOUBLE, i * 0.1);
				collector.setValue("MyVariable", Bindings.DOUBLE, data[i]);
				collector.endStep();
			}
	
			// Verify data
			StreamAccessor aa = historian.openStream(hi.id, "r");
			try {
				Assertions.assertEquals(data3.length, aa.size());
				Binding sampleBinding = Bindings.getBinding( format.format );
				// Read and assert the sample
				Object sample = aa.get(0, sampleBinding);
				ValueBand se = new ValueBand( sampleBinding, sample );
				for (int i=0; i<data3.length; i++) {
					aa.get(i, sampleBinding, se.getSample());
					Assertions.assertEquals(data3[i], se.getValue(Bindings.BYTE));
				}
			} finally {
				aa.close();
			}
			
		} finally {
			collector.close();
		}
		
	}
	
	@Test
	public void testCSVData() throws Exception {
		SamplingFormat format = simple.clone();
		SubscriptionItem hi = SubscriptionItem.createItem("MyVariable", "MySubscription", format );
		historian.create( hi );
		collector.addItem( hi );
		try {
			// Write data
			double[] data = data1;
			for (int i=0; i<data.length; i++) 
			{
				collector.beginStep(Bindings.DOUBLE, i * 0.1);
				collector.setValue("MyVariable", Bindings.DOUBLE, data[i]);
				collector.endStep();
			}
		} finally {
			collector.close();
		}
		
		CSVFormatter csv = new CSVFormatter();
		csv.addItem(historian, hi.id, "MyVariable", "MyVariable", "m");
		csv.setTimeRange(0.9, 1.5);
		csv.setTimeStep(0.1);
		StringBuilder sb = new StringBuilder();
//		csv.formulate1(new ProgressMonitor.Stub(), sb);
//		sb.append("\n----\n");
		csv.formulate2(new ProgressMonitor.Stub(), sb);
		System.out.println(sb);
		
	}
	
	@Test
	public void testExport() throws Exception {
		SamplingFormat format = byteformat.clone();
		
		// 2. Write data
		SubscriptionItem hi = SubscriptionItem.createItem("MyVariable", "MySubscription", byteformat );
		historian.create( hi );
		collector.addItem( hi );
		try {
			// Write data
			double[] data = data2;
			for (int i=0; i<data.length; i++) 
			{
				collector.beginStep(Bindings.DOUBLE, i * 0.1);
				collector.setValue("MyVariable", Bindings.DOUBLE, data[i]);
				collector.endStep();
			}			
		} finally {
			collector.close();
		}
		
		// Export data
		ByteArrayOutputStream os = new ByteArrayOutputStream(1024);
		History.exportHistory(historian, Bindings.DOUBLE, -Double.MAX_VALUE, Double.MAX_VALUE, os);
	
		// Create another historian
		HistoryManager historian2 = History.createMemoryHistory();
		ByteArrayInputStream is = new ByteArrayInputStream( os.toByteArray() );
		History.importHistory(historian2, is);
		
		// Verify data
		StreamAccessor aa = historian2.openStream(hi.id, "r");
		try {
			Assertions.assertEquals(data3.length, aa.size());
			Binding sampleBinding = Bindings.getBinding( format.format );
			// Read and assert the sample
			Object sample = aa.get(0, sampleBinding);
			ValueBand se = new ValueBand( sampleBinding, sample );
			for (int i=0; i<data3.length; i++) {
				aa.get(i, sampleBinding, se.getSample());
				Assertions.assertEquals(data3[i], se.getValue(Bindings.BYTE));
			}
		} finally {
			aa.close();
		}
	}
	
	
	@Test
	public void testStreamIterator() throws Exception {
		// 1. Create history and collector item
		SamplingFormat format = allfields.clone();
		format.deadband = 0.05;
		SubscriptionItem hi = SubscriptionItem.createItem("MyVariable", "MySubscription", format);
		historian.create(hi);
		
		// 2. Create collector
		collector.addItem(hi);
		try {
			// Add values, dt=0.1
			double[] data = data1;
			for (int i=0; i<data.length; i++) 
			{
				collector.beginStep(Bindings.DOUBLE, i * 0.1);
				collector.setValue("MyVariable", Bindings.DOUBLE, data[i]);
				collector.endStep();
			}
		} finally {
			collector.close();
		}

		// 3. Verify data
		StreamAccessor aa = historian.openStream(hi.id, "r");
		try {
			StreamIterator si = new StreamIterator( aa );
			while (si.hasNext()) {
				si.next();
				System.out.println( si );
			}
			
			si.gotoTime( -1 );
			System.out.println( si );
			
			si.gotoTime( 0.55 );
			System.out.println( si );
			
			si.proceedToTime( 0.6 );
			System.out.println( si );
			
			si.proceedToTime( 1.11 );
			System.out.println( si );

			si.gotoTime( 1.4 );
			System.out.println( si );
			
			si.gotoTime( 1.5 );
			System.out.println( si );
			
			
		} finally {
			aa.close();
		}
	}
	
}
