/*******************************************************************************
 * Copyright (c) 2007, 2024 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - GitLab #1092
 *******************************************************************************/
package org.simantics.history.impl;

import static java.nio.file.StandardWatchEventKinds.ENTRY_CREATE;
import static java.nio.file.StandardWatchEventKinds.ENTRY_DELETE;
import static java.nio.file.StandardWatchEventKinds.ENTRY_MODIFY;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.nio.file.Path;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.util.ArrayList;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Logger;

import org.simantics.databoard.Accessors;
import org.simantics.databoard.Bindings;
import org.simantics.databoard.accessor.ArrayAccessor;
import org.simantics.databoard.accessor.StreamAccessor;
import org.simantics.databoard.accessor.error.AccessorConstructionException;
import org.simantics.databoard.accessor.error.AccessorException;
import org.simantics.databoard.adapter.AdaptException;
import org.simantics.databoard.adapter.Adapter;
import org.simantics.databoard.adapter.AdapterConstructionException;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.error.RuntimeBindingConstructionException;
import org.simantics.databoard.serialization.Serializer;
import org.simantics.databoard.serialization.SerializerConstructionException;
import org.simantics.databoard.type.ArrayType;
import org.simantics.databoard.type.Component;
import org.simantics.databoard.type.Datatype;
import org.simantics.databoard.type.NumberType;
import org.simantics.databoard.type.RecordType;
import org.simantics.databoard.util.Bean;
import org.simantics.databoard.util.ObjectUtils;
import org.simantics.databoard.util.URIUtil;
import org.simantics.databoard.util.binary.BinaryMemory;
import org.simantics.databoard.util.binary.ByteBufferWriteable;
import org.simantics.history.HistoryException;
import org.simantics.history.HistoryManager;
import org.simantics.history.ItemManager;
import org.simantics.utils.FileUtils;

/**
 * File history uses workarea (directory) to manage items.
 * There are two files for every item: stream file and metadata file.
 * Metadata file is JSON ascii file.  
 *   itemname.data
 *   itemname.txt
 *  
 * @author toni.kalajainen
 *
 */
public class FileHistory implements HistoryManager {

	private static final boolean PROFILE = false;
	private static final boolean DEBUG = false;

	/** Logger */
	static Logger logger = Logger.getLogger(FileHistory.class.getName());
	
	static FilenameFilter txtFilter;
	
	static ArrayType INDEX_TYPE = Bindings.LONG_ARRAY.type();
	
	File workarea;
	Path workareaPath;
	
	/** Support async usage of the data */
	public boolean asyncUsage = true;

	private transient Bean[] cachedItems;
	private transient ItemManager cachedItemManager;
	private transient WatchService watcher;
	private transient WatchKey watchKey;
	private transient ScheduledFuture<?> pollerFuture;

	public FileHistory(File workarea) {
		this.workarea = workarea;
		this.workareaPath = workarea.toPath();
	}
	
    public File getWorkarea() {
    	return workarea;
    }
	
	@Override
	public void create(Bean... items) throws HistoryException {
		boolean changed = false;
		try {
			for (Bean item : items) {
				if (DEBUG)
					System.out.println("create(" + item + ")");
//				if ( !item.getFieldBinding("format").type().equals( Bindings.getBindingUnchecked(Datatype.class).type() ) )
//					System.err.println("Error");
				
				// Write meta data
				writeMetadata( item );
				
				// Create stream file
				String id = (String) item.getField("id");
				File dataFile = toDataFile( id );
				dataFile.createNewFile();
				
				Datatype type = (Datatype) item.getField("format");
				if ( isVariableWidth(type) ) {
					File indexFile = toIndexFile( id );
					indexFile.createNewFile();
				}
//				if ( !dataFile.createNewFile() ) {
//					throw new HistoryException("Could not create file "+dataFile);
//				}
				changed = true;
			}
		} catch (BindingException e) {
			throw new HistoryException(e);
		} catch (IOException e) {
			throw new HistoryException(e);
		} finally {
			if (changed)
				invalidateCaches();
		}
	}

	@Override
	public void delete(String... itemIds) throws HistoryException {
		boolean changed = false;
		try {
			for (String itemId : itemIds ) {
				File meta = toMetaFile( itemId );
				File data = toDataFile( itemId );
				File index = toIndexFile( itemId );
				if ( meta.exists() ) {
					if ( !meta.delete() ) {
						throw new HistoryException("Failed to delete "+meta);
					}
					changed = true;
				}
				if ( data.exists() ) {
					if ( !data.delete() ) {
						throw new HistoryException("Failed to delete "+data);
					}
					changed = true;
				}
				if ( index.exists() ) {
					if ( !index.delete() );
					else changed = true;
				}
			}
		} finally {
			if (changed)
				invalidateCaches();
		}
	}

	@Override
	public void modify(Bean... items) throws HistoryException {
		boolean metadataChanged = false;
		try {
			for ( Bean item : items ) {
				if (DEBUG)
					System.out.println("modify(" + item + ")");
//				if ( !item.getFieldBinding("format").type().equals( Bindings.getBindingUnchecked(Datatype.class).type() ) )
//					System.err.println("Error");

				String id = (String) item.getField("id");
				File metaFile = toMetaFile( id );
				if ( !metaFile.exists() ) {
					create( item );
				} else {
					Bean oldItem = getItem( id );
					File dataFile = toDataFile( id );
					if ( dataFile.exists() ) {
						boolean enabled = item.hasField("enabled") ? (Boolean) item.getFieldUnchecked("enabled") : true;
						Datatype oldFormat = (Datatype) oldItem.getField( "format" );
						Datatype newFormat = (Datatype) item.getField( "format" );
						if (DEBUG)
							System.out.println("formats: " + oldFormat +  " -> " + newFormat);
						Datatype unitStrippedOldFormat = stripUnitAnnotations(oldFormat);
						Datatype unitStrippedNewFormat = stripUnitAnnotations(newFormat);
						if (DEBUG)
							System.out.println("formats after unit strip: " + unitStrippedOldFormat +  " -> " + unitStrippedNewFormat);
						if ( enabled && !unitStrippedOldFormat.equals(unitStrippedNewFormat) ) {
							try {
								Binding oldBinding = Bindings.getBeanBinding(unitStrippedOldFormat);
								Binding newBinding = Bindings.getBeanBinding(unitStrippedNewFormat);
								Serializer oldS = Bindings.getSerializer(oldBinding);
								Serializer newS = Bindings.getSerializer(newBinding);
								if (oldS.getConstantSize()==null || newS.getConstantSize()==null || oldS.getConstantSize()!=newS.getConstantSize())
									throw new HistoryException("Changing of file format is not supported to: "+dataFile);
								Adapter adapter = Bindings.getAdapter(oldBinding, newBinding);
								Object oldSample = oldBinding.createDefault();
								Object newSample = newBinding.createDefault();
								StreamAccessor sa = openStream(id, "rw");
								try {
									int c = sa.size();
									for (int i=0; i<c; i++) {
										sa.get(i, oldBinding, oldSample);
										newSample = adapter.adapt(oldSample);
										sa.set(i, newBinding, newSample);
									}
								} finally {
									sa.close();
								}
							} catch (AdapterConstructionException e) {
								throw new HistoryException("Changing of file format is not supported to: "+id);
							} catch (SerializerConstructionException e) {
								throw new HistoryException("Changing of file format is not supported to: "+id);
							} catch (AccessorException e) {
								throw new HistoryException("Changing of file format failed to: "+id);
							} catch (AdaptException e) {
								throw new HistoryException("Changing of file format failed to: "+id);
							}
						}
					} else {
						dataFile.createNewFile();
					}

					// Write new meta-data if necessary
					if (!equalsWithoutState(item, oldItem)) {
						writeMetadata( item );
						metadataChanged = true;
					}
				}
			}
		} catch (BindingException e) {
			throw new HistoryException( e );
		} catch (IOException e) {
			throw new HistoryException( e );
		} finally {
			if (metadataChanged)
				invalidateCaches();
		}
	}

	@Override
	public Bean getItem(String itemId) throws HistoryException {
		return getItem( toMetaFile( itemId ) );
	}
	
	void writeMetadata( Bean item ) throws HistoryException {
		
//		long s = System.nanoTime();
		try {
			String id = (String) item.getField("id");
			String idEnc  = URIUtil.encodeURI(id);
			File metaFile = new File(workarea, idEnc + ".txt");
			Serializer typeSerializer = Bindings.getSerializer( Bindings.getBindingUnchecked(Datatype.class) );
			Serializer beanSerializer = Bindings.getSerializer( item.getBinding() );
			int size = typeSerializer.getSize(item.getBinding().type()) +
					beanSerializer.getSize(item);
			byte data[] = new byte[size];
			DataOutput out = new ByteBufferWriteable( ByteBuffer.wrap(data) );
			
			if ( metaFile.exists() && asyncUsage ) {
				if (DEBUG)
					System.out.println("WARNING: FileHistory.writeMetadata: on SLOW path for " + item);
				File tmpFile = new File(workarea, idEnc + ".tmp");
				File tmp2File = new File(workarea, idEnc + ".tmp2");
				tmpFile.delete();
	
				typeSerializer.serialize(out, item.getBinding().type());
				beanSerializer.serialize(out, item);
				FileUtils.writeFile(tmpFile, data);
				metaFile.renameTo(tmp2File);
				tmpFile.renameTo(metaFile);
				tmp2File.delete();
			} else {
				typeSerializer.serialize(out, item.getBinding().type());
				beanSerializer.serialize(out, item);
				FileUtils.writeFile(metaFile, data);
			}

//			if (PROFILE)
//				System.out.println("PROFILE: FileHistory.writeMetadata( " + metaFile.getName() + " ) in " + ((System.nanoTime() - s)*1e-6) + " ms");
		} catch (BindingException e) {
			throw new HistoryException(e);
		} catch (IOException e) {
			throw new HistoryException(e);
		} catch (SerializerConstructionException e) {
			throw new HistoryException(e);
		}
	}

	Bean getItem(File file) throws HistoryException {
		try {
			byte[] data = FileUtils.readFile(file);
			DataInput in = new BinaryMemory(data);  
			Serializer typeSerializer = Bindings.getSerializer( Bindings.getBindingUnchecked(Datatype.class) );			
			Datatype type = (Datatype) typeSerializer.deserialize(in);
			Binding beanBinding = Bindings.getBeanBinding( type );
			Serializer s = Bindings.getSerializer( beanBinding );
			Bean bean = (Bean) s.deserialize(in);
			return bean;
		} catch(BufferUnderflowException e) {
			throw new HistoryException( e );
		} catch(IOException e) {
			throw new HistoryException( e );
//		} catch (DataTypeSyntaxError e) {
//			throw new HistoryException( e );
		} catch (RuntimeBindingConstructionException e) {
			throw new HistoryException( e );
		} catch (SerializerConstructionException e) {
			throw new HistoryException( e );
		}
	}
	
	File toMetaFile(String itemId)
	{
		String name = URIUtil.encodeURI(itemId) + ".txt";
		File f = new File(workarea, name);
		return f;
	}

	File toDataFile(String itemId)
	{
		String name = URIUtil.encodeURI(itemId) + ".data";
		File f = new File(workarea, name);
		return f;
	}
	
	File toIndexFile(String itemId)
	{
		String name = URIUtil.encodeURI(itemId) + ".index";
		File f = new File(workarea, name);
		return f;
	}
	
	boolean isVariableWidth(Datatype type) 
	throws HistoryException {
		try {
			Binding beanBinding = Bindings.getBeanBinding(type);
			Serializer s = Bindings.getSerializer( beanBinding );
			return s.getConstantSize() == null;
		} catch (SerializerConstructionException e) {
			throw new HistoryException(e);
		}
	}

	private static final WatchEvent.Kind<?>[] WATCH_KINDS = {
			StandardWatchEventKinds.ENTRY_CREATE,
			StandardWatchEventKinds.ENTRY_DELETE,
			StandardWatchEventKinds.ENTRY_MODIFY,
	};

	private static final WatchEvent.Modifier[] WATCH_MODIFIERS = {};

	@Override
	public Bean[] getItems() throws HistoryException {
		synchronized (this) {
			Bean[] c = cachedItems;
			if (c != null)
				return c;

			var result = new ArrayList<>();
			File[] files = workarea.listFiles(txtFilter);
			if ( files != null ) {
				for (File file : files) {
					result.add( getItem(file) );
				}
			}

			var ret = result.toArray( new Bean[ result.size() ] );

			// Only cache the result if it is possible to watch the data source for changes
			if (startMetadataWatching()) {
				cachedItems = ret;
			}

			return ret;
		}
	}

	@Override
	public ItemManager getItemManager() throws HistoryException {
		synchronized (this) {
			var im = cachedItemManager;
			if (im != null)
				return im;

			im = cachedItemManager = new ItemManager(getItems());
			return im;
		}
	}

	Runnable pollWatcher = () -> {
		synchronized (FileHistory.this) {
			var wk = watcher.poll();
			if (wk != null) {
				for (WatchEvent<?> event : wk.pollEvents()) {
					WatchEvent.Kind<?> kind = event.kind();

					// Context for directory entry event is the file name of entry
					@SuppressWarnings("unchecked")
					WatchEvent<Path> evt = (WatchEvent<Path>) event;

					Path name = evt.context();
					if (name != null) {
						var fn = name.getFileName();
						if (fn != null) {
							var lcfn = fn.toString().toLowerCase();
							if (lcfn.endsWith(".txt")) {
								if (kind == ENTRY_CREATE || kind == ENTRY_MODIFY || kind == ENTRY_DELETE) {
									stopMetadataWatching();
									invalidateCaches();
									break;
								}
							}
						}
					}
				}
			}
		}
	};

	private boolean startMetadataWatching() {
		// Try to start watcher to invalidate the cache upon need
		try {
			watcher = workareaPath.getFileSystem().newWatchService();
			watchKey = workareaPath.register(watcher, WATCH_KINDS, WATCH_MODIFIERS);
			pollerFuture = getPollingExecutor().scheduleWithFixedDelay(pollWatcher, 100, 100, TimeUnit.MILLISECONDS);
			return true;
		} catch (IOException e) {
			// Ok, no watching and no caching then.
			return false;
		}
	}

	private void stopMetadataWatching() {
		if (pollerFuture != null) {
			pollerFuture.cancel(false);
			pollerFuture = null;
		}
		if (watchKey != null) {
			watchKey.cancel();
			watchKey = null;
		}
		if (watcher != null) {
			FileUtils.uncheckedClose(watcher);
			watcher = null;
		}
	}

	@Override
	public synchronized void close() {
		stopMetadataWatching();
		invalidateCaches();
	}

	private synchronized void invalidateCaches() {
		cachedItems = null;
		cachedItemManager = null;
	}

	@Override
	public StreamAccessor openStream(String itemId, String mode) throws HistoryException {
		try {
			Bean bean = getItem(itemId);
			Datatype format = (Datatype) bean.getField("format");
			ArrayType arrayType = new ArrayType(format);
			File dataFile = toDataFile( itemId );
			if ( isVariableWidth(format) ) {
				File indexFile = toIndexFile( itemId );
				ArrayAccessor index = Accessors.openStream(indexFile, INDEX_TYPE, mode);
				return (StreamAccessor) Accessors.openStream(dataFile, arrayType, mode, index);
			} else {
				return (StreamAccessor) Accessors.openStream(dataFile, arrayType, mode);
			}
		} catch (AccessorConstructionException e) {
			throw new HistoryException(e);
		} catch (BindingException e) {
			throw new HistoryException(e);
		}
	}

	@Override
	public boolean exists(String itemId) throws HistoryException {
		return toMetaFile(itemId).exists();
	}
	
	@Override
	public int hashCode() {
		return workarea.hashCode();
	}
	
	@Override
	public boolean equals(Object obj) {
		if ( obj==null ) return false;
		if ( obj instanceof FileHistory == false ) return false;
		FileHistory other = (FileHistory) obj;		
		return other.workarea.equals(workarea);
	}
	
	@Override
	public String toString() {
		return "FileHistory: "+workarea;
	}

	private boolean equalsWithoutState(Bean i1, Bean i2) {
		Component[] components1 = i1.getBinding().type().getComponents();
		Component[] components2 = i2.getBinding().type().getComponents();
		int components = Math.min(components1.length, components2.length);
		for (int c = 0; c < components; ++c) {
			Object o1 = i1.getFieldUnchecked(c);
			Object o2 = i2.getFieldUnchecked(c);
			if ("collectorState".equals(components1[c].name) && (o1 == null || o2 == null))
				continue;
			if (!ObjectUtils.objectEquals(o1, o2))
				return false;
		}
		return true;
	}

	static {
		txtFilter = new FilenameFilter() {
			public boolean accept(File dir, String name) {
				return name.toLowerCase().endsWith(".txt");
			}
		};
	}

	private static Datatype stripUnitAnnotations(Datatype datatype) {
		if (datatype instanceof NumberType) {
			NumberType nt = (NumberType) datatype;
			if (nt.getUnit() != null) {
				Binding dtb = Bindings.getBindingUnchecked(Datatype.class);
				datatype = nt = (NumberType) Bindings.cloneUnchecked(datatype, dtb, dtb);
				nt.setUnit(null);
			}
		} else if (datatype instanceof ArrayType) {
			ArrayType at = (ArrayType) datatype;
			Datatype ct = at.componentType();
			Datatype component = stripUnitAnnotations(ct);
			if (component != ct) {
				Binding dtb = Bindings.getBindingUnchecked(Datatype.class);
				datatype = at = (ArrayType) Bindings.cloneUnchecked(datatype, dtb, dtb);
				at.setComponentType(component);
			}
		} else if (datatype instanceof RecordType) {
			RecordType rt = (RecordType) datatype;
			int componentCount = rt.getComponentCount();
			Component[] newComponents = new Component[componentCount];
			for (int i = 0; i < componentCount; ++i) {
				Component c = rt.getComponent(i);
				Datatype ct = c.type;
				Datatype sct = stripUnitAnnotations(ct);
				newComponents[i] = new Component(c.name, sct);
			}
			return new RecordType(rt.isReferable(), newComponents);
		}
		return datatype;
	}

	private static ScheduledExecutorService POLLER;

	public static synchronized ScheduledExecutorService getPollingExecutor() {
		if (POLLER == null) {
			final ThreadGroup tg = new ThreadGroup("FileHistory");
			final AtomicInteger counter = new AtomicInteger(0);
			ThreadFactory tf = new ThreadFactory() {
				@Override
				public Thread newThread(Runnable r) {
					Thread t = new Thread(tg, r, "FileHistory-Metadata-Poller-"+(counter.incrementAndGet()));
					if (!t.isDaemon())
						t.setDaemon(true);
					if (t.getPriority() != Thread.NORM_PRIORITY)
						t.setPriority(Thread.NORM_PRIORITY);
					return t;
				}
			};
			POLLER = new ScheduledThreadPoolExecutor( 0, tf );
		}
		return POLLER;
	}

}
