/*******************************************************************************
 * Copyright (c) 2007, 2017 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - #7313
 *******************************************************************************/
package org.simantics.charts.ui;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.viewers.ISelection;
import org.simantics.Simantics;
import org.simantics.charts.Activator;
import org.simantics.charts.internal.VariableUtils;
import org.simantics.charts.ontology.ChartResource;
import org.simantics.charts.query.AddChartItem;
import org.simantics.charts.query.ChartItemDescriptor;
import org.simantics.databoard.Bindings;
import org.simantics.databoard.type.BooleanType;
import org.simantics.databoard.type.Datatype;
import org.simantics.databoard.type.NumberType;
import org.simantics.databoard.util.ObjectUtils;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.procedure.adapter.ProcedureAdapter;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.SelectionHints;
import org.simantics.db.layer0.adapter.DropActionFactory;
import org.simantics.db.layer0.request.PossibleModel;
import org.simantics.db.layer0.variable.RVI;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.layer0.variable.VariableReference;
import org.simantics.db.layer0.variable.Variables.Role;
import org.simantics.modeling.ModelingResources;
import org.simantics.modeling.PropertyVariables;
import org.simantics.modeling.PropertyVariablesImpl;
import org.simantics.modeling.utils.VariableReferences;
import org.simantics.trend.configuration.TrendItem.Renderer;
import org.simantics.ui.selection.WorkbenchSelectionElement;
import org.simantics.utils.ui.ErrorLogger;
import org.simantics.utils.ui.ISelectionUtils;

/**
 * @author Tuukka Lehtonen
 */
public class ChartDropActionFactory implements DropActionFactory {

    @Override
    public Runnable create(ReadGraph g, Object target, Object source, int operation) throws DatabaseException {
        //System.out.println("CHART DROP: " + source + " -> " + target);

        final Resource chart = ISelectionUtils.getSinglePossibleKey(target, SelectionHints.KEY_MAIN, Resource.class);
        if (chart == null) return null;
        Resource targetModel = g.syncRequest(new PossibleModel(chart));
        if (targetModel == null) return null;

        if(source instanceof RVI) {
            List<VariableReference> refs = Collections.singletonList(new VariableReference((RVI)source,
                    VariableUtils.getDatatype(g, targetModel, (RVI) source), null));
            return new AddVariableToChartAction(chart, null, refs).init(g);
        }

        List<PropertyVariables> vars = ISelectionUtils.getPossibleKeys(source, SelectionHints.KEY_MAIN, PropertyVariables.class);
        if (!vars.isEmpty()) {
            // FIXME: this is a hack for indexed value support
            vars = PropertyVariablesImpl.resolve(g, vars);
            List<VariableReference> references = toPropertyReferences(g, targetModel, vars);
            if (!references.isEmpty())
                return new AddVariableToChartAction(chart, null, references).init(g);
        }

        List<Variable> vars2 = ISelectionUtils.getPossibleKeys(source, SelectionHints.KEY_MAIN, Variable.class);
        if (!vars2.isEmpty()) {
            List<VariableReference> references = toReferences(g, targetModel, vars2);
            if (!references.isEmpty())
                return new AddVariableToChartAction(chart, null, references).init(g);
        }

        if(source instanceof ISelection) {
            List<WorkbenchSelectionElement> wses = ISelectionUtils.filterSelection((ISelection)source, WorkbenchSelectionElement.class);
            if (!wses.isEmpty()) {
                List<Variable> wsevars = new ArrayList<>();
                ChartVariable av = new ChartVariable(g);
                for(WorkbenchSelectionElement wse : wses) {
                    Variable v = wse.getContent(av);
                    if(v != null)
                        wsevars.add(v);
                }

                List<VariableReference> references = toReferences(g, targetModel, wsevars);
                if (!wsevars.isEmpty()) {
                    return new AddVariableToChartAction(chart, null, references).init(g);
                }
            }
        }
        {
            List<Resource> srcs = ISelectionUtils.getPossibleKeys(source, SelectionHints.KEY_MAIN, Resource.class);
            ModelingResources MOD = ModelingResources.getInstance(g);
            ChartResource CHART = ChartResource.getInstance(g);
            List<ChartItemDescriptor> newItems = new ArrayList<ChartItemDescriptor>();
            Set<Resource> movedPlots = new HashSet<Resource>();
            for (Resource res : srcs) {
                if (g.isInstanceOf(res, MOD.Subscription_Item)) {
                    Resource model = g.syncRequest(new PossibleModel(res));
                    if (ObjectUtils.objectEquals(targetModel, model)) {
                        ChartItemDescriptor desc = new ChartItemDescriptor();
                        desc.subscriptionItem = res;

                        Datatype datatype = g.getPossibleRelatedValue(res, MOD.Subscription_Item_Datatype, Bindings.getBindingUnchecked(Datatype.class));
                        desc.renderer = datatype instanceof BooleanType ? Renderer.Binary : Renderer.Analog;

                        newItems.add(desc);
                    }
                } else if (g.isInstanceOf(res, CHART.Chart_Item)) {
                    Resource model = g.syncRequest(new PossibleModel(res));
                    if (ObjectUtils.objectEquals(targetModel, model))
                        movedPlots.add(res);
                }
            }
            if (!newItems.isEmpty() || !movedPlots.isEmpty())
                return addPlots(chart, newItems, movedPlots);
        }

        if (source instanceof String) {
            return handleStringDrop(g, chart, targetModel, (String) source);
        }

        return null;
    }

    private Runnable handleStringDrop(ReadGraph graph, Resource chart, Resource targetModel, String source) {
        try {
            List<VariableReference> refs = filterReferences(VariableUtils.getVariableReferencesFromString(graph, targetModel, source));
            return refs != null && !refs.isEmpty() ? new AddVariableToChartAction(chart, null, refs).init(graph) : null;
        } catch (DatabaseException e) {
            Activator.getDefault().getLog().log(new Status(IStatus.WARNING, Activator.PLUGIN_ID, getClass().getSimpleName() + ": Unrecognized String input: " + source));
            return null;
        }
    }

	private static List<VariableReference> toReferences(ReadGraph graph, Resource contextIndexRoot, List<Variable> variables) throws DatabaseException {
        if (variables.isEmpty())
            return Collections.emptyList();
        return filterReferences( graph.syncRequest(VariableReferences.variablesToReferences(contextIndexRoot, variables)) );
    }

    private static List<VariableReference> toPropertyReferences(ReadGraph graph, Resource contextIndexRoot, List<PropertyVariables> variables) throws DatabaseException {
        if (variables.isEmpty())
            return Collections.emptyList();
        return filterReferences( graph.syncRequest(VariableReferences.toReferences(contextIndexRoot, variables)) );
    }

    private static List<VariableReference> filterReferences(List<VariableReference> variables) throws DatabaseException {
        return variables.stream()
                // Must be a property or substructure of one to even potentially have a value
                .filter(ref -> ref.containsPartWithRole(Role.PROPERTY))
                .filter(ref -> ref.datatype instanceof BooleanType || ref.datatype instanceof NumberType)
                .collect(Collectors.toList());
    }

    public static Runnable addPlots(Resource chart, List<ChartItemDescriptor> references, Set<Resource> movedPlots) {
        return () -> {
            Simantics.getSession().asyncRequest(
                    AddChartItem.addAndMoveChartItems(chart, references, movedPlots),
                    new ProcedureAdapter<Collection<Resource>>() {
                        @Override
                        public void exception(Throwable e) {
                            if (e != null)
                                ErrorLogger.defaultLogError(e);
                        }
                    });
        };
    }

}