package org.simantics.browsing.ui.nattable;

import java.util.Collection;
import java.util.List;
import java.util.regex.Pattern;

import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.nebula.widgets.nattable.NatTable;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.simantics.browsing.ui.GraphExplorer;
import org.simantics.utils.ui.AdaptionUtils;

/**
 * Selects tree items based on pressed key events.<p>
 * 
 * 
 * @author Marko Luukkainen <marko.luukkainen@vtt.fi>
 *
 */
public class KeyToSelectionAdapter extends KeyAdapter {

	private static final int KEY_INPUT_DELAY = 500;
	
    private final NatTableGraphExplorer          explorer;

    private String matcher = "";
    private int prevEvent = 0;
    private int columns = 0;
        
    protected Pattern alphaNum;
    
    /**
     * @param contextProvider
     * @param explorer
     */
    public KeyToSelectionAdapter(GraphExplorer explorer) {
    	this(explorer,"\\p{Alnum}");
    }
    
    public KeyToSelectionAdapter(GraphExplorer explorer, String pattern) {
        assert explorer != null;

        this.explorer = (NatTableGraphExplorer)explorer;
        this.alphaNum = Pattern.compile(pattern);
    }
    
    public boolean acceptKey(char key) {
    	return alphaNum.matcher(Character.toString(key)).matches();
    }
    
    @Override
    public void keyPressed(KeyEvent e) {
    	if (explorer.isDisposed())
            return;

         if (!acceptKey(e.character))
         	 return;
         // concatenate / replace matcher.
         if ((e.time - prevEvent) > KEY_INPUT_DELAY )
    		 matcher = "";
    	 prevEvent = e.time;
    	 matcher = matcher += Character.toString(e.character);

    	 NatTable tree = explorer.getControl();
    	 columns = explorer.getColumns().length;
    	 
    	 IStructuredSelection sel = (IStructuredSelection)explorer.getWidgetSelection();
    	 Collection<RowSelectionItem> selected = AdaptionUtils.adaptToCollection(sel, RowSelectionItem.class);
    	 
    	  
    	 TreeNode item = find(tree, selected);
    	 
    	 if (item == null && matcher.length() > 1) {
    		 matcher = matcher.substring(matcher.length()-1);
    		 item = find(tree, selected);
    	 }
    	 
    	 if (item != null) {
    		 explorer.show(item);
    		 explorer.select(item);
		 } 
    	 // without this the default handling would take over.
    	 e.doit = false;
    }
    
    private TreeNode previous = null;
    private boolean foundPrev = false;
    
	private TreeNode find(NatTable tree, Collection<RowSelectionItem> selected) {
		TreeNode item = null;

		List<TreeNode> items = explorer.getItems();
		
		if (selected.size() == 0) {
			previous = null;
			foundPrev = true;
			item = findItem(items);

		} else {
			previous = selected.iterator().next().item;
			foundPrev = false;
			item = findItem(items);
			if (item == null) {
				previous = null;
				foundPrev = true;
				item = findItem(items);
			}
		}
		return item;
	}
    
    private TreeNode findItem(List<TreeNode> items) {
    	for (int i = 0; i < items.size(); i++) {
    		TreeNode item = items.get(i);
    		if (item != previous) {
    			if (foundPrev && matches(item, columns, matcher))
    				return item;
	    		
    		} else {
    			foundPrev = true;
    		}
    	}
    	return null;
    }
    
    
    /**
     * 
     * @param item
     * @param depth Depth of the item in the tree.
     * @param columns Number of columns.
     * @param string Matching string.
     * @return
     */
    protected boolean matches(TreeNode item, int columns, String matcher) {
    	for (int c = 0; c < columns; c++) {	
			if (matchesColumn(item, c, matcher)) {
				return true;
			}
		}
    	return false;
    }
    
    /**
     * 
     * @param item
     * @param column
     * @param matcher
     * @return
     */
    protected boolean matchesColumn(TreeNode item, int column, String matcher) {
    	String text = item.getValueString(column);
    	if (text == null)
    		return false;
		if (text.toLowerCase().startsWith(matcher)) {
			return true;
		}
		return false;
    }

}
