/*******************************************************************************
 * Copyright (c) 2015 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.model.actions;

import org.simantics.browsing.ui.BuiltinKeys;
import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.model.nodetypes.NodeType;
import org.simantics.browsing.ui.model.nodetypes.NodeTypeMultiMap;
import org.simantics.browsing.ui.model.tests.Test;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.utils.Logger;
import org.simantics.db.exception.DatabaseException;
import org.simantics.viewpoint.ontology.ViewpointResource;

/**
 * Encapsulation of a browse context {@link NodeType} and input tester. Consists
 * of a node type, a priority and a possible {@link Test} that returns whether
 * the action related to this test is enabled or not. If there is no test
 * defined, the test is considered to return <code>true</code>.
 * 
 * @author Tuukka Lehtonen
 */
public class TestContribution implements Comparable<TestContribution> {

    NodeType nodeType;
    Test test;
    double priority;

    public TestContribution(NodeType nodeType, Test test, double priority) {
        super();
        this.nodeType = nodeType;
        this.test = test;
        this.priority = priority;
    }

    public static void load(ReadGraph g, Resource r, NodeTypeMultiMap<TestContribution> contributions) throws DatabaseException {
        ViewpointResource VR = ViewpointResource.getInstance(g);

        NodeType nodeType = g.adapt(g.getSingleObject(r, VR.TestContribution_HasNodeType), NodeType.class);

        Resource testResource = g.getPossibleObject(r, VR.TestContribution_HasTest);
        Test test = testResource == null ? null : g.adapt(testResource, Test.class);

        Double mpriority = g.getPossibleRelatedValue(r, VR.DropActionContribution_HasPriority);
        double priority = mpriority == null ? 0.0 : mpriority.doubleValue();

        contributions.put(nodeType, new TestContribution(nodeType, test, priority));
    }

    public boolean test(ReadGraph graph, NodeContext context) {
        try {
            return test == null || test.test(graph, context.getConstant(BuiltinKeys.INPUT));
        } catch (DatabaseException e) {
            Logger.defaultLogError(e);
            return false;
        }
    }

    @Override
    public int compareTo(TestContribution o) {
        return Double.compare(o.priority, priority);
    }

}
