/*******************************************************************************
 * Copyright (c) 2023, 2024 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.acorn;

import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

import org.rocksdb.RocksDB;
import org.rocksdb.RocksDBException;
import org.rocksdb.RocksIterator;
import org.simantics.db.IO;

/**
 * @author Jani Simomaa
 * @author Tuukka Lehtonen
 */
public class RocksIO implements IO {

	private static final Charset ENC = StandardCharsets.UTF_8;

	private RocksDB db;
	private byte[] path;
	private int writePosition = 0;

	public RocksIO(RocksDB db, AcornKeyRocks path) {
		this.db = db;
		this.path = path.keyBytes();
	}

	@Override
	public synchronized int saveBytes(byte[] bytes, int length, boolean overwrite) throws IOException {
		try {
			if (overwrite) {
				writePosition = 0;
				if(length != bytes.length) {
					db.put(path, Arrays.copyOf(bytes, length));
				} else {
					db.put(path, bytes);
				}
			} else {
				String key = new String(path, ENC) + "." + writePosition;
				if(length != bytes.length) {
					db.put(key.getBytes(ENC), Arrays.copyOf(bytes, length));
				} else {
					db.put(key.getBytes(ENC), bytes);
				}
			}
			int result = writePosition;
			writePosition += length;
			return result;
		} catch (RocksDBException e) {
			e.printStackTrace();
			return 0;
		}
	}

	public synchronized byte[] readFully() throws RocksDBException {
		RocksIterator it = db.newIterator();
		it.seek(path);
		String prefix = new String(path, ENC);
		List<Map.Entry<Integer,byte[]>> data = new ArrayList<>();
		int subPos = prefix.length()+1;
		int count = 0;
		while(it.isValid()) {
			String key = new String(it.key(), ENC);
			if(!key.startsWith(prefix))
				break;
			int index = 0;
			if(key.length() > subPos)
				index = Integer.parseInt(key.substring(subPos));
			byte[] bytes = it.value();
			data.add(Map.entry(index, bytes));
			count += bytes.length;
			//System.err.println(" -" + key + " " + index + " " + bytes.length);
			it.next();
		}
		if(data.size() == 0)
			return null;
		if(data.size() == 1)
			return data.iterator().next().getValue();
		byte[] result = new byte[count];
		for(Map.Entry<Integer,byte[]> entry : data) {
			byte[] part = entry.getValue();
			System.arraycopy(part, 0, result, entry.getKey(), part.length);
		}
		return result;
	}

	@Override
	public synchronized byte[] readBytes(long offset, int length) throws IOException {
		try {
			byte[] bs = readFully();
			if(bs == null)
				throw new IOException("No data");
			if(bs.length == length)
				return bs;
			byte[] result = new byte[length];
			System.arraycopy(bs, (int)offset, result, 0, length);
			return result;
		} catch (RocksDBException e) {
			throw new IOException(e);
		}
	}

	public synchronized long length() throws IOException {
		RocksIterator it = db.newIterator();
		it.seek(path);
		String prefix = new String(path, ENC);
		int count = 0;
		while(it.isValid()) {
			String key = new String(it.key());
			if(!key.startsWith(prefix))
				break;
			byte[] bytes = it.value();
			count += bytes.length;
			it.next();
		}
		return count;
	}

}
