/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.debug.ui;

import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Collections;
import java.util.LinkedList;
import java.util.Set;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.browser.Browser;
import org.eclipse.swt.custom.CTabFolder;
import org.eclipse.swt.custom.CTabItem;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbenchSite;
import org.simantics.Simantics;
import org.simantics.browsing.ui.swt.widgets.impl.WidgetSupport;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.debug.ListenerReport;
import org.simantics.db.exception.CancelTransactionException;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.management.ISessionContext;
import org.simantics.db.service.DebugSupport;
import org.simantics.debug.ui.internal.Activator;
import org.simantics.ui.workbench.IPropertyPage;
import org.simantics.utils.ui.LayoutUtils;
import org.simantics.utils.ui.SWTUtils;
import org.simantics.views.swt.SimanticsView;

@SuppressWarnings("deprecation")
public class SessionDebuggerView extends SimanticsView {

    public static final String VIEW_ID = "org.simantics.debug.sessionDebugger"; //$NON-NLS-1$

    private CTabFolder folder;
    private Text commandLine;
    private Browser console;

    private final LinkedList<String> terminal = new LinkedList<String>();
    private final LinkedList<String> history = new LinkedList<String>();
    private int historyPosition = -1;

    @Override
    protected Set<String> getBrowseContexts() {
        return Collections.singleton(""); //$NON-NLS-1$
    }

    private CTabItem createItem(int index, CTabFolder folder, Control control) {
        CTabItem item = new CTabItem(folder, SWT.NONE, index);
        item.setControl(control);
        return item;
    }

    @Override
    protected void createControls(Composite body, IWorkbenchSite site, ISessionContext context, WidgetSupport support) {

        body.setLayout(LayoutUtils.createNoBorderGridLayout(1));

        folder = new CTabFolder(body, SWT.BOTTOM | SWT.FLAT);
        folder.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));

        Composite shell = new Composite(folder, SWT.NONE);
        shell.setLayout(LayoutUtils.createNoBorderGridLayout(1));

        commandLine = new Text(shell, SWT.BORDER);
        commandLine.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false));

        console = new Browser(shell, SWT.BORDER);
        console.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));

        commandLine.addKeyListener(new KeyAdapter() {
            @Override
            public void keyReleased(KeyEvent e) {
                if (e.keyCode == SWT.ARROW_UP) {
                    if (historyPosition > 10 || historyPosition >= (history.size() - 1)) {
                        return;
                    }
                    commandLine.setText(history.get(++historyPosition));
                } else if (e.keyCode == SWT.ARROW_DOWN) {
                    if (historyPosition < 0) {
                        return;
                    } else if (historyPosition == 0) {
                        commandLine.setText(""); //$NON-NLS-1$
                        historyPosition = -1;
                    } else {
                        commandLine.setText(history.get(--historyPosition));
                    }
                } else if (e.keyCode == SWT.ESC) {
                    historyPosition = -1;
                    commandLine.setText(""); //$NON-NLS-1$
                } else if (e.keyCode == SWT.CR || e.keyCode == SWT.KEYPAD_CR) {
                    applyCommand(commandLine.getText());
                }
            }
        });

        CTabItem shellItem = createItem(0, folder, shell);
        shellItem.setText(Messages.SessionDebuggerView_Shell);

//        SessionDebugger deprecated = new SessionDebugger(folder, SWT.NONE, Simantics.getSession());
//        deprecated.initializeUI();
//        CTabItem debuggerItem = createItem(1, folder, deprecated);
//        debuggerItem.setText("Debugger");

        folder.setSelection(shellItem);
    }

    private void applyCommand(final String command) {
        try {
            Simantics.getSession().syncRequest(new WriteRequest() {
                @Override
                public void perform(WriteGraph graph) throws DatabaseException {
                    DebugSupport support = graph.getService(DebugSupport.class);
                    Object data = support.query(graph, command);
                    apply(command, data);
                }
            });
        } catch (CancelTransactionException e) {
        } catch (DatabaseException e) {
            Activator.getDefault().getLog().log(
                    new Status(IStatus.ERROR, Activator.PLUGIN_ID,
                            Messages.SessionDebuggerView_ActivatorUnexpectedException + command, e));
        }
    }

    private void addHistory(String command, Object output) {
        if (output instanceof String) {
            terminal.addFirst((String) output);
        } else if (output instanceof Path) {
            try {
                Path p = (Path) output;
                long size = Files.size(p);
                if (size < (1L << 16)) {
                    terminal.addFirst(new String(Files.readAllBytes(p), "UTF-8")); //$NON-NLS-1$
                }
                terminal.addFirst(NLS.bind(Messages.SessionDebuggerView_WroteCommand ,new Object[] { command  , p})); 
            } catch (IOException e) {
                Activator.getDefault().getLog().log(
                        new Status(IStatus.ERROR, Activator.PLUGIN_ID,
                                NLS.bind(Messages.SessionDebuggerView_ActivatorUnexpectedIOException , new Object[]{ command, e})));
            }
        } else {
            throw new IllegalArgumentException("Unsupported output argument type " + output); //$NON-NLS-1$
        }
        if (terminal.size() > 10)
            terminal.removeLast();

        history.addFirst(command);
        if (history.size() > 10)
            history.removeLast();

        historyPosition = -1;
    }

    private void apply(String command, Object data) {
        if (data instanceof String) {
            SWTUtils.asyncExec(commandLine, () -> {
                commandLine.setText(""); //$NON-NLS-1$
                addHistory(command, data);
                console.setText(formatTerminal());
            });
        } else if (data instanceof ListenerReport) {
            SWTUtils.asyncExec(commandLine, () -> {
                try {
                    addHistory(command, dumpListenerReport((ListenerReport) data));
                    commandLine.setText(""); //$NON-NLS-1$
                    console.setText(formatTerminal());
                } catch (IOException e) {
                    Activator.getDefault().getLog().log(
                            new Status(IStatus.ERROR, Activator.PLUGIN_ID,
                                    Messages.SessionDebuggerView_ActivatorUnexpectedIOException + command, e));
                }
            });
        }
    }

    private Path dumpListenerReport(ListenerReport data) throws IOException {
        File f = Simantics.getTempfile("debug", "listenerReport"); //$NON-NLS-1$ //$NON-NLS-2$
        try (PrintStream out = new PrintStream(f, "UTF-8")) { //$NON-NLS-1$
            out.print("<pre>"); //$NON-NLS-1$
            data.print(out);
            out.print("</pre>"); //$NON-NLS-1$
        }
        return f.toPath();
    }

    private String formatTerminal() {
        StringBuilder b = new StringBuilder();
        b.append("<html><head/><body>\n"); //$NON-NLS-1$
        for (String s : terminal)
            b.append(s).append("<br/>\n"); //$NON-NLS-1$
        b.append("</body></html>"); //$NON-NLS-1$
        return b.toString();
    }

    @Override
    public void setFocus() {
        folder.setFocus();
    }

    @Override
    protected IPropertyPage getPropertyPage() {
        return null;
    }

}
