package org.simantics.browsing.ui.nattable;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.eclipse.nebula.widgets.nattable.config.CellConfigAttributes;
import org.eclipse.nebula.widgets.nattable.config.ConfigRegistry;
import org.eclipse.nebula.widgets.nattable.config.IConfigRegistry;
import org.eclipse.nebula.widgets.nattable.layer.cell.ILayerCell;
import org.eclipse.nebula.widgets.nattable.painter.cell.CellPainterWrapper;
import org.eclipse.nebula.widgets.nattable.painter.cell.ICellPainter;
import org.eclipse.nebula.widgets.nattable.style.CellStyleAttributes;
import org.eclipse.nebula.widgets.nattable.style.ConfigAttribute;
import org.eclipse.nebula.widgets.nattable.style.DisplayMode;
import org.eclipse.nebula.widgets.nattable.style.IDisplayModeOrdering;
import org.eclipse.nebula.widgets.nattable.style.IStyle;
import org.eclipse.nebula.widgets.nattable.style.Style;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Rectangle;

public class GEStyler extends CellPainterWrapper {

	private GETreeData treeData;

	public GEStyler(GETreeData treeData, ICellPainter painter) {
		super(painter);
		this.treeData = treeData;
	}

	private ConfigRegistryWrapper wrapper = new ConfigRegistryWrapper();

	@Override
	public void paintCell(ILayerCell cell, GC gc, Rectangle rectangle, IConfigRegistry configRegistry) {
		wrapper.clear();
		wrapper.wrappedRegistry = configRegistry;
		TreeNode node = treeData.getDataAtIndex(cell.getRowIndex());
		PubStyle style = new PubStyle();
		node.getStyle(cell.getColumnIndex(), style);
		Image image = node.getImage(cell.getColumnIndex());
		if (image != null)
			style.setAttributeValue(CellStyleAttributes.IMAGE, image);

		{
			// Attempt to reuse existing style, so that we override only those parts that GE contributions define.
			Style defStyle = getStyle(DisplayMode.NORMAL, "BODY");
			style.applyTo(defStyle);
			wrapper.setSpecificConfigAttribute(CellConfigAttributes.CELL_STYLE, DisplayMode.NORMAL, "BODY", defStyle);
		}
		{
			Style defStyle = getStyle(DisplayMode.SELECT, "BODY");
			style.applyTo(defStyle);
			wrapper.setSpecificConfigAttribute(CellConfigAttributes.CELL_STYLE, DisplayMode.SELECT, "BODY", defStyle);
		}
		{
			Style defStyle = getStyle(DisplayMode.SELECT_HOVER, "BODY");
			style.applyTo(defStyle);
			wrapper.setSpecificConfigAttribute(CellConfigAttributes.CELL_STYLE, DisplayMode.SELECT_HOVER, "BODY", defStyle);
		}
		if (configRegistry.getSpecificConfigAttribute(CellConfigAttributes.CELL_STYLE, DisplayMode.NORMAL, "ODD_BODY") != null) {
			Style oddStyle = getStyle(DisplayMode.NORMAL, "ODD_BODY");
			style.applyTo(oddStyle);
			wrapper.setSpecificConfigAttribute(CellConfigAttributes.CELL_STYLE, DisplayMode.NORMAL, "ODD_BODY", oddStyle);
		}
		if (configRegistry.getSpecificConfigAttribute(CellConfigAttributes.CELL_STYLE, DisplayMode.NORMAL, "EVEN_BODY") != null) {
			Style evenStyle = getStyle(DisplayMode.NORMAL, "EVEN_BODY");
			style.applyTo(evenStyle);
			wrapper.setSpecificConfigAttribute(CellConfigAttributes.CELL_STYLE, DisplayMode.NORMAL, "EVEN_BODY", evenStyle);
		}
		super.paintCell(cell, gc, rectangle, wrapper);
	}

	private static class PubStyle implements IStyle {
		Map<ConfigAttribute<?>, Object> styleAttributeValueMap = new HashMap<>();

		@Override
		@SuppressWarnings("unchecked")
		public <T> T getAttributeValue(ConfigAttribute<T> styleAttribute) {
			return (T) this.styleAttributeValueMap.get(styleAttribute);
		}

		@Override
		public <T> void setAttributeValue(ConfigAttribute<T> styleAttribute, T value) {
			this.styleAttributeValueMap.put(styleAttribute, value);
		}

		@Override
		public String toString() {
			StringBuilder resultBuilder = new StringBuilder();
			resultBuilder.append(this.getClass().getSimpleName() + ": "); //$NON-NLS-1$

			for (Entry<ConfigAttribute<?>, Object> entry : this.styleAttributeValueMap.entrySet()) {
				resultBuilder.append(entry.getKey()
						+ ": " + entry.getValue() + "\n"); //$NON-NLS-1$ //$NON-NLS-2$
			}

			return resultBuilder.toString();
		}

		public void applyTo(IStyle style) {
			for (Entry<ConfigAttribute<?>, Object> e : styleAttributeValueMap.entrySet()) {
				style.setAttributeValue((ConfigAttribute)e.getKey(), e.getValue());
			}
		}

	}

	private Style getStyle(DisplayMode displayMode, String configLabel)  {
		Style s  = (Style)wrapper.getSpecificConfigAttribute(CellConfigAttributes.CELL_STYLE, displayMode, configLabel);
		if (s == null)
			s = new Style();
		else 
			s = new Style(s);
		return s;
	}

	private class ConfigRegistryWrapper extends ConfigRegistry {
		IConfigRegistry wrappedRegistry;
		Map<ConfigAttribute<?>, Map<DisplayMode, Map<String, ?>>> configRegistry = new HashMap<>();

		public void clear() {
			configRegistry.clear();
		}

		@Override
		public <T> T getConfigAttribute(ConfigAttribute<T> configAttribute, DisplayMode targetDisplayMode,
				String... configLabels) {
			return wrappedRegistry.getConfigAttribute(configAttribute, targetDisplayMode, configLabels);
		}

		@Override
		public <T> T getConfigAttribute(ConfigAttribute<T> configAttribute, DisplayMode targetDisplayMode,
				List<String> configLabels) {
			return wrappedRegistry.getConfigAttribute(configAttribute, targetDisplayMode, configLabels);
		}

		@Override
		public <T> T getSpecificConfigAttribute(ConfigAttribute<T> configAttribute, DisplayMode displayMode,
				String configLabel) {
			T value = _getSpecificConfigAttribute(configAttribute, displayMode, configLabel);
			if (value != null)
				return value;
			return wrappedRegistry.getSpecificConfigAttribute(configAttribute, displayMode, configLabel);
		}

		public <T> T _getSpecificConfigAttribute(ConfigAttribute<T> configAttribute,
				DisplayMode displayMode, String configLabel) {
			T attributeValue = null;

			Map<DisplayMode, Map<String, ?>> displayModeConfigAttributeMap = this.configRegistry.get(configAttribute);
			if (displayModeConfigAttributeMap != null) {
				Map<String, T> configAttributeMap = (Map<String, T>) displayModeConfigAttributeMap.get(displayMode);
				if (configAttributeMap != null) {
					attributeValue = configAttributeMap.get(configLabel);
					if (attributeValue != null) {
						return attributeValue;
					}
				}
			}

			return attributeValue;
		}

		public <T> void setSpecificConfigAttribute(ConfigAttribute<T> configAttribute, DisplayMode displayMode,
				String configLabel, T attributeValue) {
			Map<DisplayMode, Map<String, ?>> displayModeConfigAttributeMap = this.configRegistry.get(configAttribute);
			if (displayModeConfigAttributeMap == null) {
				displayModeConfigAttributeMap = new HashMap<>();
				this.configRegistry.put(configAttribute, displayModeConfigAttributeMap);
			}

			Map<String, T> configAttributeMap = (Map<String, T>) displayModeConfigAttributeMap.get(displayMode);
			if (configAttributeMap == null) {
				configAttributeMap = new HashMap<>();
				displayModeConfigAttributeMap.put(displayMode, configAttributeMap);
			}

			configAttributeMap.put(configLabel, attributeValue);
		}

		@Override
		public <T> void registerConfigAttribute(ConfigAttribute<T> configAttribute, T attributeValue) {
			wrappedRegistry.registerConfigAttribute(configAttribute, attributeValue);
		}

		@Override
		public <T> void registerConfigAttribute(ConfigAttribute<T> configAttribute, T attributeValue,
				DisplayMode targetDisplayMode) {
			wrappedRegistry.registerConfigAttribute(configAttribute, attributeValue, targetDisplayMode);
		}

		@Override
		public <T> void registerConfigAttribute(ConfigAttribute<T> configAttribute, T attributeValue,
				DisplayMode targetDisplayMode, String configLabel) {
			wrappedRegistry.registerConfigAttribute(configAttribute, attributeValue, targetDisplayMode, configLabel);
		}

		@Override
		public <T> void unregisterConfigAttribute(ConfigAttribute<T> configAttributeType) {
			wrappedRegistry.unregisterConfigAttribute(configAttributeType);
		}

		@Override
		public <T> void unregisterConfigAttribute(ConfigAttribute<T> configAttributeType, DisplayMode displayMode) {
			wrappedRegistry.unregisterConfigAttribute(configAttributeType, displayMode);
		}

		@Override
		public <T> void unregisterConfigAttribute(ConfigAttribute<T> configAttributeType, DisplayMode displayMode,
				String configLabel) {
			wrappedRegistry.unregisterConfigAttribute(configAttributeType, displayMode, configLabel);
		}

		@Override
		public IDisplayModeOrdering getDisplayModeOrdering() {
			return wrappedRegistry.getDisplayModeOrdering();
		}

	}
}
