/*******************************************************************************
 * Copyright (c) 2019, 2023 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.document.server.request;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.simantics.db.AsyncReadGraph;
import org.simantics.db.ReadGraph;
import org.simantics.db.common.procedure.adapter.TransientCacheAsyncListener;
import org.simantics.db.common.request.UnaryAsyncRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.request.VariableRead;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.procedure.AsyncProcedure;
import org.simantics.document.server.DocumentServerUtils.AttributesRequest;
import org.simantics.document.server.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DocumentRequest extends VariableRead<List<JSONObject>> {

	private static final Logger LOGGER = LoggerFactory.getLogger(DocumentRequest.class);

	public static boolean PROFILE = false;
	// Thresholds in microseconds
	public static int PROFILE_THRESHOLD_NODEREQUEST = 2000;
	public static int PROFILE_THRESHOLD_VALUEREQUEST = 500;

	public DocumentRequest(Variable var) {
		super(var);
	}

	static class CollectNodesRequest extends UnaryAsyncRead<Collection<Variable>, Collection<JSONObject>> {

		public CollectNodesRequest(Collection<Variable> nodes) {
			super(nodes);
		}

		@Override
		public void perform(AsyncReadGraph graph, AsyncProcedure<Collection<JSONObject>> procedure) {
			HashSet<JSONObject> rs = new HashSet<JSONObject>(parameter.size()); // result

			for(Variable node : parameter) {
				graph.asyncRequest(new AttributesRequest(node), new AsyncProcedure<JSONObject> () {

					@Override
					public void execute(AsyncReadGraph graph, JSONObject result) {
						synchronized(rs) {
							rs.add(result);
						}
					}

					@Override
					public void exception(AsyncReadGraph graph, Throwable throwable) {
					}

				});

			}
			procedure.execute(graph, rs);

		}

	}

	@Override
	public List<JSONObject> perform(ReadGraph graph) throws DatabaseException {

		long s = System.nanoTime();

		Set<Variable> nodes = graph.syncRequest(new NodesRequest(variable), TransientCacheAsyncListener.<Set<Variable>>instance());
		if(nodes.isEmpty()) {
			return Collections.emptyList();
		}

		if(PROFILE) {
			long dura = System.nanoTime()-s;
			LOGGER.info("DocumentRequest loaded " + nodes.size() + " nodes in " + 1e-6*dura + "ms. " + variable.getURI(graph));
		}

		Collection<JSONObject> rs = graph.syncRequest(new CollectNodesRequest(nodes));

		if(PROFILE) {
			long dura = System.nanoTime()-s;
			LOGGER.info("DocumentRequest loaded attributes in " + 1e-6*dura + "ms. " + variable.getURI(graph));
		}

		ArrayList<JSONObject> result = new ArrayList<JSONObject>(rs);
		Collections.sort(result, new Comparator<JSONObject>() {

			@Override
			public int compare(JSONObject o1, JSONObject o2) {
				return o1.id.compareTo(o2.id);
			}

		});

		return result;

	}
}