/*******************************************************************************
 * Copyright (c) 2007, 2023 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.document.server;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import org.simantics.db.AsyncReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ResourceAsyncRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.ServiceException;
import org.simantics.db.layer0.request.PropertyInfo;
import org.simantics.db.layer0.request.PropertyInfoRequest;
import org.simantics.db.procedure.AsyncProcedure;
import org.simantics.document.base.ontology.DocumentationResource;

public class DocumentPropertyKeys extends ResourceAsyncRead<List<String>> {

	public DocumentPropertyKeys(Resource resource) {
		super(resource);
	}

	@Override
	public void perform(AsyncReadGraph graph, final AsyncProcedure<List<String>> procedure) {

		final List<String> result = new ArrayList<>();

		DocumentationResource DOC = DocumentationResource.getInstance(graph);

		try {
			if(graph.hasStatement(resource, DOC.Properties_dataDefinitions))
				result.add("dataDefinitions");
		} catch(ServiceException e) {
			Functions.LOGGER.info(e.getMessage(), e);
		}

		graph.forEachDirectPredicate(resource, new AsyncProcedure<Set<Resource>>() {

			@Override
			public void execute(AsyncReadGraph graph, Set<Resource> predicates) {

				for(Resource predicate : predicates) {

					try {

						PropertyInfo info = graph.syncRequest(new PropertyInfoRequest(predicate));

						if(info.isHasProperty && info.hasClassification(DocumentationResource.URIs.Document_AttributeRelation)) {
							result.add(info.name);
						} else {
							Resource definition = graph.getPossibleObject(predicate, DOC.Document_definesAttributeRelation);
							if(definition != null) {
								PropertyInfo info2 = graph.syncRequest(new PropertyInfoRequest(definition));
								result.add(info2.name);
							}
						}

					} catch (DatabaseException e) {
						Functions.LOGGER.info(e.getMessage(), e);
					}

				}

				procedure.execute(graph, result);

			}

			@Override
			public void exception(AsyncReadGraph graph, Throwable throwable) {
				Functions.LOGGER.info(throwable.getMessage(), throwable);
				procedure.exception(graph, throwable);
			}

		});

	}

}