package org.simantics.utils;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.StandardOpenOption;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.CopyOnWriteArrayList;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.adapter.AdaptException;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.mutable.Variant;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Development {

	private static final Logger LOGGER = LoggerFactory.getLogger(Development.class);
	public static TreeMap<String,Integer> histogram = new TreeMap<>();

	public static final boolean DEVELOPMENT = false;
	
	public static final String PRINT = "Development.print";

	public static String LOG_FILE = "development.log";
	public static String REPORT_FILE = "development.report";

	private static String breakLine; 

	static OutputStream log;

	static {

		if (DEVELOPMENT) {
			try {
				if(Files.exists(new File(LOG_FILE).toPath()))
					developmentClear();
				log = new BufferedOutputStream(new FileOutputStream(LOG_FILE));
			} catch (FileNotFoundException e) {
				LOGGER.error("Error while initializing Development files", e);
			} catch (IOException e) {
				LOGGER.error("Error while initializing Development files", e);
			}
		}

	}

	public static void breakOnLine() {
		new Exception("Development breakLine breakpoint was hit.").printStackTrace();
	}

	public static void log(String line) {
		if (DEVELOPMENT) {
			try {
				if(breakLine != null && breakLine.equals(line))
					breakOnLine();
				if(line.length() > 500) line = line.substring(0, 500);
				log.write((line+'\n').getBytes());
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
	}

	public static void developmentSetBreakLine(String line) {
		breakLine = line;
	}

	public static void developmentClearBreakLine(String line) {
		breakLine = null;
	}

	public static void developmentClear() throws IOException {
		Files.write(new File(LOG_FILE).toPath(), new byte[0], StandardOpenOption.TRUNCATE_EXISTING);
	}

	public static String developmentReportPR() throws IOException {
		return developmentReportPrefix("PR ");
	}

	public static String developmentReportPQ() throws IOException {
		return developmentReportPrefix("PQ ");
	}

	public static String developmentReportPrefix(String prefix) throws IOException {
		Map<String,Integer> histogram = new HashMap<>();
		Files.lines(new File(LOG_FILE).toPath()).forEach(line -> {
			if(line.startsWith(prefix)) {
				String req = line.substring(2);
				Integer i = histogram.get(req);
				histogram.put(req, i == null ? 1 : i+1);
			}
		});
		StringBuilder sb = new StringBuilder(1024*16);
		histogram.entrySet().stream()
		.sorted((e1, e2) -> {
			int i = -Integer.compare(e1.getValue(), e2.getValue());
			if (i != 0)
				return i;
			return e1.getKey().compareTo(e2.getKey());
		})
		.forEach(e -> {
			sb.append(e.getValue());
			sb.append(' ');
			sb.append(e.getKey());
			sb.append('\n');
		});
		FileUtils.writeFile(new File(REPORT_FILE), sb.toString().getBytes());
		return sb.toString();
	}

	public interface DevelopmentListener {
		
		void handle(Object event);
		
	}
	
	final static private HashMap<String, Variant> properties = new HashMap<>(); 
	
	final static private CopyOnWriteArrayList<DevelopmentListener> listeners = new CopyOnWriteArrayList<>();

	static {
		
		if(DEVELOPMENT) {
			listeners.add(new DevelopmentListener() {
	
				@Override
				public void handle(Object event) {
					if((Boolean) getProperty(PRINT, Bindings.BOOLEAN))
						LOGGER.info(event.toString());
				}
	
			});
		}

	}
	
	public static void addListener(DevelopmentListener listener) {
		listeners.add(listener);
	}
	
	public static void removeListener(DevelopmentListener listener) {
		listeners.remove(listener);
	}
	
	public static void dispatchEvent(Object event) {
		for(DevelopmentListener listener : listeners) {
			listener.handle(event);
		}
	}
	
	
	public static void setProperty(String name, Object value, Binding binding) {
		assert(name != null);
		assert(binding != null);
		try {
			binding.assertInstaceIsValid(value);
		} catch (BindingException e) {
			throw new RuntimeException(e);
		}
		properties.put(name, new Variant(binding, value));
	}

	@SuppressWarnings("unchecked")
	public static <T> T getProperty(String name, Binding binding) {
		Variant value = properties.get(name);
		if(value == null) return null;
		try {
			return (T)value.getValue(binding);
		} catch (AdaptException e) {
			throw new RuntimeException(e);
		}
	}
	
	public static boolean isTrue(String name) {
		Boolean value = getProperty(name, Bindings.BOOLEAN);
		return value != null && value;
	}
	
	public static Map<String, Variant> getProperties() {
		return properties;
	}
	
	public static void recordHistogram(Object o) {
		String key = o.toString();
		Integer i = histogram.get(key);
		histogram.put(key, i == null ? 1 : i+1);
	}
	
}
