/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.ui.dnd;

import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.DragSource;
import org.eclipse.swt.dnd.DragSourceEvent;
import org.eclipse.swt.dnd.DragSourceListener;
import org.eclipse.swt.dnd.TextTransfer;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.widgets.Control;
import org.simantics.db.Session;
import org.simantics.db.common.ResourceArray;
import org.simantics.db.exception.DatabaseException;
import org.simantics.ui.utils.ResourceAdaptionUtils;

/**
 * A basic combined implementation of an SWT {@link DragSource} and
 * {@link DragSourceListener} that supports the following transfers:
 * <ul>
 * <li>{@link TextTransfer}</li>
 * <li>{@link ResourceReferenceTransfer}</li>
 * <li>{@link LocalObjectTransfer}</li>
 * </ul>
 * 
 * <p>
 * Initialization requires the SWT {@link Control} which the drag source is
 * attached, an {@link ISelectionProvider} for the control and a database
 * session.
 * 
 * @author Tuukka Lehtonen
 */
public class BasicDragSource implements DragSourceListener, SessionContainer {

    private Transfer[]           transferAgents;

    private ISelectionProvider   selectionProvider;

    private IStructuredSelection sel;

    private ResourceArray[]      resources;

    private Session              session;

    private String               purpose;

    /**
     * @param selectionProvider
     * @param sourceControl
     * @param session the database Session to be used for serialization by this
     *        drag source or <code>null</code> to put this drag source in
     *        disabled state at construction time. To later enable this drag
     *        source, use {@link #setSession(Session)} to set the resource
     *        serializer.
     */
    public BasicDragSource(ISelectionProvider selectionProvider, Control sourceControl, Session session) {
        this(selectionProvider, sourceControl, session, null);
    }

    /**
     * @param selectionProvider
     * @param sourceControl
     * @param session the database Session to be used for serialization by this
     *        drag source or <code>null</code> to put this drag source in
     *        disabled state at construction time. To later enable this drag
     *        source, use {@link #setSession(Session)} to set the resource
     *        serializer.
     * @param purpose a string for defining a designation of the purpose of this
     *        drag source. <code>null</code> indicates no special purpose.
     */
    public BasicDragSource(ISelectionProvider selectionProvider, Control sourceControl, Session session, String purpose) {
        this.selectionProvider = selectionProvider;
        this.transferAgents = new Transfer[] { 
                TextTransfer.getInstance(), 
                ResourceReferenceTransfer.createInstance(purpose),
                LocalObjectTransfer.getTransfer()
        };
        this.session = session;
        this.purpose = purpose;
        DragSource source = new DragSource(sourceControl, DND.DROP_LINK | DND.DROP_MOVE | DND.DROP_COPY | DND.DROP_DEFAULT);
        source.setTransfer(transferAgents);
        source.addDragListener(this);

        // NOTE: This will disable SWT DND selection screenshot painting.
        source.setDragSourceEffect(new NoImageDragSourceEffect(sourceControl));
    }

    @Override
    public Session getSession() {
        return session;
    }

    @Override
    public void setSession(Session session) {
        this.session = session;
    }

    public void dragStart(DragSourceEvent event) {
        // Don't start dragging by default.
        event.doit = false;

        // Drag won't work without a database session.
        if (session == null)
            return;

        // Allow drag to start only if the selection is non-empty.
        sel = (IStructuredSelection) selectionProvider.getSelection();
        if (sel == null || sel.isEmpty())
            return;

        resources = ResourceAdaptionUtils.toResourceArrays(sel);

        event.doit = resources.length > 0; 
    }

    public void dragSetData(DragSourceEvent event) {
        if (ResourceReferenceTransfer.getInstance().isSupportedType(event.dataType)) {
            event.data = resources;  
        } else if (LocalObjectTransfer.getTransfer().isSupportedType(event.dataType)) {
            event.data = sel;
        } else if (TextTransfer.getInstance().isSupportedType(event.dataType)) {
            try {
                event.data = ResourceTransferUtils.createStringTransferable(session, resources, purpose);
            } catch (DatabaseException e) {
                event.doit = false;
                e.printStackTrace();
            }
        }
    }

    public void dragFinished(DragSourceEvent event) {
        // release resources
        sel = null;
        resources = null;
    }

}
