/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.ui.dialogs;

import org.eclipse.jface.action.IAction;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.resource.LocalResourceManager;
import org.eclipse.jface.resource.ResourceManager;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;
import org.simantics.ui.internal.Activator;
import org.simantics.utils.strings.AlphanumComparator;
import org.simantics.utils.ui.action.IPriorityAction;


/**
 * @author Tuukka Lehtonen
 */
public class ActionChooserDialog extends Dialog {

    private static final String DIALOG = "ActionChooserDialog"; //$NON-NLS-1$

    private final IAction[]     actions;

    private final String        title;

    private final String        description;

    private IAction             chosenAction;

    private TableViewer         viewer;

    private IDialogSettings     dialogBoundsSettings;

    private ResourceManager     resourceManager;

    public ActionChooserDialog(Shell parent, IPriorityAction[] actions, String title, String description) {
        super(parent);
        this.actions = actions;
        this.title = title;
        this.description = description;

        IDialogSettings settings = Activator.getDefault().getDialogSettings();
        dialogBoundsSettings = settings.getSection(DIALOG);
        if (dialogBoundsSettings == null)
            dialogBoundsSettings = settings.addNewSection(DIALOG);
    }

    @Override
    protected IDialogSettings getDialogBoundsSettings() {
        return dialogBoundsSettings;
    }

    @Override
    protected void configureShell(Shell newShell) {
        if (title != null) {
            newShell.setText(title);
        } else {
            newShell.setText("Choose Action");
        }
        super.configureShell(newShell);
    }

    @Override
    protected int getShellStyle() {
        return super.getShellStyle() | SWT.RESIZE;
    }

    @Override
    protected Point getInitialSize() {
        Point defaultSize = getShell().computeSize(SWT.DEFAULT, SWT.DEFAULT, true);
        Point result = super.getInitialSize();
        if (defaultSize.equals(result))
            return new Point(300, 300);
        return result;
    }

    @Override
    protected Control createDialogArea(Composite parent) {
        Composite composite = (Composite) super.createDialogArea(parent);

        this.resourceManager = new LocalResourceManager(JFaceResources.getResources());
        composite.addListener(SWT.Dispose, new Listener() {
            @Override
            public void handleEvent(Event event) {
                resourceManager.dispose();
                resourceManager = null;
            }
        });

        if (description != null) {
            Label label = new Label(composite, 0);
            label.setText(description);
            label.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false));
        }

        viewer = new TableViewer(composite, SWT.V_SCROLL | SWT.H_SCROLL | SWT.BORDER | SWT.SINGLE);
        viewer.getControl().setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
        viewer.setContentProvider(new ArrayContentProvider());
        viewer.setLabelProvider(new ActionLabelProvider());
        viewer.setSorter(sorter);
        viewer.setInput(actions);

        viewer.addDoubleClickListener(new IDoubleClickListener() {
            @Override
            public void doubleClick(DoubleClickEvent event) {
                okPressed();
            }
        });

        viewer.addSelectionChangedListener(new ISelectionChangedListener() {
            @Override
            public void selectionChanged(SelectionChangedEvent event) {
                ActionChooserDialog.this.selectionChanged(event.getSelection());
            }
        });

        if (actions.length > 0) {
            viewer.setSelection(new StructuredSelection(actions[0]), true);
        }

        applyDialogFont(composite);
        return composite;
    }

    private void selectionChanged(ISelection s) {
        Button ok = getButton(IDialogConstants.OK_ID);
        IStructuredSelection iss = (IStructuredSelection) s;
        if (iss == null || iss.isEmpty()) {
            if (ok != null)
                ok.setEnabled(false);
            return;
        }

        if (ok != null)
            ok.setEnabled(true);
        return;
    }

    @Override
    protected void okPressed() {
        chosenAction = (IAction) ((IStructuredSelection) viewer.getSelection()).getFirstElement();
        super.okPressed();
    }

    public IAction getChosenAction() {
        return chosenAction;
    }

    private final ViewerSorter sorter = new ViewerSorter() {
        @Override
        public int category(Object element) {
            // Sort actions in descending priority order
            if (element instanceof IPriorityAction) {
                IPriorityAction action = (IPriorityAction) element;
                return -action.getPriority();
            }
            return Integer.MAX_VALUE;
        }
        @Override
        public int compare(Viewer viewer, Object e1, Object e2) {
            IPriorityAction a1 = (IPriorityAction) e1;
            IPriorityAction a2 = (IPriorityAction) e2;
            return AlphanumComparator.CASE_INSENSITIVE_COMPARATOR.compare(a1.getText(), a2.getText());
        }
    };

    class ActionLabelProvider extends LabelProvider {
        @Override
        public Image getImage(Object element) {
            IAction a = (IAction) element;
            ImageDescriptor desc = a.getImageDescriptor();
            if(desc == null) return null;
            else return resourceManager.createImage(desc);
        }

        @Override
        public String getText(Object element) {
            IPriorityAction a = (IPriorityAction) element;
            return a.getText() + " (" + a.getPriority() + ")";
//            return a.getText();
        }
    }

}
