package org.simantics.spreadsheet.solver;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Consumer;

import org.simantics.spreadsheet.Range;
import org.simantics.spreadsheet.SpreadsheetVisitor;
import org.simantics.spreadsheet.Spreadsheets;

import it.unimi.dsi.fastutil.objects.ObjectArrayList;

@SuppressWarnings("rawtypes")
public class SpreadsheetLine implements SpreadsheetElement<SpreadsheetCell, SpreadsheetLines>, SheetNode {

    private static final long serialVersionUID = -304574098117404663L;

    final private SpreadsheetLines parent;
    final public int row;
    int id;

    public ObjectArrayList<SpreadsheetCell> cells = new ObjectArrayList<>();

    public SpreadsheetLine(SpreadsheetLines parent, int row) {
        this.parent = parent;
        this.row = row;
        this.id = getEngine().getBook().getNewId(this);
    }

    public int getId() {
        return id;
    }

    public SpreadsheetEngine getEngine() {
        return ((SpreadsheetLines)parent).getEngine();
    }
    
    public int getRow() {
        return row;
    }

    public SpreadsheetLine possibleOffset(int offset) {
        return getEngine().getLine(row+offset);
    }
    
    public SpreadsheetCell cellAt(int column) {
        int index = Collections.binarySearch(cells, new BinarySearch(column), new Comparator<BinarySearch>() {
            @Override
            public int compare(BinarySearch bs1, BinarySearch bs2) {
                return Integer.compare(bs1.column, bs2.column);
            }
        });
        if(index >= 0)
            return cells.get(index);
        else
            return null;
    }

    public String getLinesPath() {
        return "/" + ((SpreadsheetLines)parent).getLinesPath() + "/" + getName();
    }

    public void forCells(Consumer<SpreadsheetCell> consumer, int min, int max) {
        for(int i=min;i<cells.size() && i<max;i++) {
            SpreadsheetCell cell = cells.get(i);
            if(SpreadsheetCell.EMPTY == cell) continue;
            consumer.accept(cell);
        }
    }

    public List<SpreadsheetCell> getCells(int min, int max) {
        ArrayList<SpreadsheetCell> result = new ArrayList<>();
        forCells(cell -> result.add(cell), min, max);
        return result;
    }

    @Override
    public String getName() {
        return "Row"+(row);
    }

    @Override
    public Map getChildren() {
        String rowName = ""+row;
        Map<String,SpreadsheetCell> result = new HashMap<>();
        for(int i=0;i<cells.size();i++) {
            SpreadsheetCell cell = cells.get(i);
            if(SpreadsheetCell.EMPTY == cell) continue;
            String name = Spreadsheets.columnName(i) + rowName;
            result.put(name, cell);
        }
        return result;
    }

    @Override
    public Map getProperties() {
        return Collections.singletonMap("typeURI", new SpreadsheetTypeNode(Spreadsheets.LINE_TYPE_URI));
    }

    Object resolve(String[] parts, int index) {

        if(index == parts.length) return this;

        Range r = Spreadsheets.decodeCellAbsolute(parts[index]);
        return cells.get(r.startColumn);

    }

    public void accept(SpreadsheetVisitor v) {
        v.visit(this);
    }

    public String getPath() {
        return ((SpreadsheetLines)parent).getPath() + "/" + getName();
    }

    @Override
    public Optional<SpreadsheetLines> getParent() {
        return Optional.of(parent);
    }

    @Override
    public List<SpreadsheetCell> getSpreadsheetChildren() {
        return cells;
    }

    @Override
    public void remove(SpreadsheetCell child) {

    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((parent == null) ? 0 : parent.hashCode());
        result = prime * result + row;
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        SpreadsheetLine other = (SpreadsheetLine) obj;
        if (parent == null) {
            if (other.parent != null)
                return false;
        } else if (!parent.equals(other.parent))
            return false;
        if (row != other.row)
            return false;
        return true;
    }

}
