/*******************************************************************************
 * Copyright (c) 2007, 2019 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - gitlab #384
 *******************************************************************************/
package org.simantics.simulation.ui;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.contexts.IContextActivation;
import org.eclipse.ui.contexts.IContextService;
import org.simantics.simulation.experiment.IDynamicExperiment;
import org.simantics.simulation.experiment.IExperiment;
import org.simantics.simulation.project.IExperimentManager;
import org.simantics.simulation.project.IExperimentManagerListener;

public class ExperimentManagerListener implements IExperimentManagerListener {

    public static final String     EXPERIMENT_CONTEXT         = "org.simantics.simulation.ui.experiment";
    public static final String     DYNAMIC_EXPERIMENT_CONTEXT = "org.simantics.simulation.ui.dynamicExperiment";

    static Set<IExperimentManager> managers = 
        new HashSet<IExperimentManager>();

    IExperimentManager manager;

    Collection<IContextActivation> contextActivations = 
        new ArrayList<IContextActivation>();

    public ExperimentManagerListener(IExperimentManager manager) {
        this.manager = manager;
    }

    public static void listenManager(IExperimentManager manager) {
        synchronized(managers) {
            if(managers.contains(manager))
                return;
            ExperimentManagerListener listener = 
                new ExperimentManagerListener(manager);
            manager.addListener(listener);
            managers.add(manager);
        }
    }

    @Override
    public void activeExperimentLoaded(final IExperiment experiment) {
        if (!PlatformUI.isWorkbenchRunning())
            return;

        experiment.addListener(new org.simantics.simulation.ui.e4.ExperimentListener());

        final IWorkbench workbench = PlatformUI.getWorkbench();
        workbench.getDisplay().asyncExec(new Runnable() {
            @Override
            public void run() {
                if (workbench.isClosing())
                    return;
                IContextService contextService =
                    (IContextService)PlatformUI.getWorkbench()
                    .getService(IContextService.class);
                synchronized(contextActivations) {
                    contextActivations.add(contextService.activateContext(EXPERIMENT_CONTEXT));
                    if(experiment instanceof IDynamicExperiment)
                        contextActivations.add(contextService.activateContext(DYNAMIC_EXPERIMENT_CONTEXT));
                }
            }
        });
    }

    @Override
    public void activeExperimentUnloaded() {
        synchronized(contextActivations) {
            final Collection<IContextActivation> oldContextActivations = 
                contextActivations;

            contextActivations = new ArrayList<IContextActivation>();

            if (!PlatformUI.isWorkbenchRunning())
                return;

            final IWorkbench workbench = PlatformUI.getWorkbench();
            workbench.getDisplay().asyncExec(new Runnable() {
                @Override
                public void run() {
                    if (workbench.isClosing())
                        return;

                    IContextService contextService =
                        (IContextService)workbench.getService(IContextService.class);
                    contextService.deactivateContexts(oldContextActivations);
                }
            });
        }
    }

    @Override
    public void managerDisposed() {
        synchronized(managers) {
            manager.removeListener(this);
            managers.remove(manager);
        }
    }

}
