package org.simantics.selectionview;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.PossibleIndexRoot;
import org.simantics.db.common.utils.Functions;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.SelectionHints;
import org.simantics.db.layer0.adapter.Instances;
import org.simantics.db.layer0.request.VariableIndexRoot;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.layer0.variable.Variables;
import org.simantics.layer0.Layer0;
import org.simantics.ui.selection.WorkbenchSelectionUtils;
import org.simantics.utils.ui.ISelectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StandardSelectionProcessor implements SelectionProcessor<Object, ReadGraph> {
    private static final Logger LOGGER = LoggerFactory.getLogger(StandardSelectionProcessor.class);

	private static Resource getIndexRoot(ReadGraph graph, Object selection) throws DatabaseException {
	    
	    if(selection instanceof Collection) {
	        
	        Collection<Object> os = (Collection<Object>)selection;
	        HashSet<Resource> roots = new HashSet<>();
	        for(Object o : os) {
	            Resource root = getIndexRoot(graph, o);
	            if(root != null)
	                roots.add(root);
	        }
	        
	        if(roots.size() == 1)
	            return roots.iterator().next();
	        else
	            return null;
	        
	    } else {

    		Variable variable = WorkbenchSelectionUtils.getPossibleVariable(graph, selection);
    		if (variable != null) {
    			return graph.sync(new VariableIndexRoot(variable));
    		}
    
    		variable = ISelectionUtils.filterSingleSelection(selection, Variable.class);
    		if (variable != null) {
    			return graph.sync(new VariableIndexRoot(variable));
    		}
    
    		Resource indexRoot = ISelectionUtils.getSinglePossibleKey(selection, SelectionHints.KEY_MODEL, Resource.class, false);
    		if (indexRoot == null) {
    			Resource r = ISelectionUtils.getSinglePossibleKey(selection, SelectionHints.KEY_MAIN, Resource.class);
    			if (r != null)
    				indexRoot = graph.sync(new PossibleIndexRoot(r));
    		}
    
    		return indexRoot != null && graph.isInstanceOf(indexRoot, Layer0.getInstance(graph).IndexRoot) ? indexRoot : null;
    		
	    }
	    
	}

	@Override
	public Collection<?> process(Object selection, ReadGraph graph) {
		return processStatic(selection, graph);
	}

	public static Collection<?> processStatic(Object selection, ReadGraph graph) {

		try {

			SelectionViewResources SEL = SelectionViewResources.getInstance(graph);
			Instances selectionProcessorFinder = graph.adapt(SEL.SelectionProcessorContribution, Instances.class);
			Instances contributionFinder = graph.adapt(SEL.TabContribution, Instances.class);
			Instances transformationFinder = graph.adapt(SEL.SelectionTransformation, Instances.class);

			ArrayList<TabContribution<?>> contributions = new ArrayList<>();

			Resource index = getIndexRoot(graph, selection);
			if(index != null) {

				for(Resource r : contributionFinder.find(graph, index)) {
					TabContribution<?> contribution = graph.adapt(r, TabContribution.class);
					contributions.add(contribution);
					if (LOGGER.isDebugEnabled())
						LOGGER.debug("-contribution " + graph.getURI(r));
				}

				ArrayList<Resource> transforms = new ArrayList<>();
				transforms.addAll(transformationFinder.find(graph, index));
				if (LOGGER.isDebugEnabled()) {
					for(Resource r : transforms)
						LOGGER.debug("-transform " + NameUtils.getSafeLabel(graph, r));
				}
				
				HashSet<Object> inputs = new HashSet<>();
				HashSet<Object> newInputs = new HashSet<>();
				inputs.add(selection);
				boolean changed = true;
				while(changed) {
					for(Resource tr : transforms) {
						for(Object input : inputs) {
							try {
								Object newInput = Functions.exec(graph, tr, graph, input);
								if(newInput != null)
									newInputs.add(newInput);
							} catch (DatabaseException e) {
							}
						}
					}
					changed = inputs.addAll(newInputs);
					newInputs.clear();
				}

				if (LOGGER.isDebugEnabled()) {
					for(Object o : inputs)
						LOGGER.debug("-input " + inputName(graph, o));
				}

				Set<ComparableTabContributor> result = new HashSet<>();
				for(TabContribution c : contributions) {
					for(Object input : inputs) {
						try {
							if (LOGGER.isDebugEnabled()) {
								LOGGER.debug("contribution: " + c);
								LOGGER.debug("->input: " + input);
								LOGGER.debug("->input name: " + inputName(graph, input));
							}
							if (c.accept(graph, input))
								c.contribute(graph, input, result);
						} catch (Exception e) {
							LOGGER.error("Selection view tab contribution failed for contribution " + c, e);
						}
					}
				}

				for(Resource r : selectionProcessorFinder.find(graph, index)) {
					if (LOGGER.isDebugEnabled())
						LOGGER.debug("-SelectionProcessor contribution " + graph.getURI(r));
					@SuppressWarnings("unchecked")
					SelectionProcessor<Object, ReadGraph> processor = graph.adapt(r, SelectionProcessor.class);
					Collection<?> contribs = processor.process(selection, graph);
					if (contribs != null && !contribs.isEmpty()) {
						for (Object contrib : contribs) {
							if (contrib instanceof ComparableTabContributor) {
								result.add((ComparableTabContributor) contrib);
							}
						}
					}
				}

				if (LOGGER.isDebugEnabled()) {
					LOGGER.debug("result: " + result);
				}

				return result;

			}

		} catch (DatabaseException e) {
			LOGGER.error("Selection view tab contribution failed unexpectedly.", e);
		}

		return Collections.emptyList();
	}

	private static String inputName(ReadGraph graph, Object o) throws DatabaseException {
		if(o instanceof Resource) {
			return "Resource: " + NameUtils.getURIOrSafeNameInternal(graph, (Resource)o);
		} else if(o instanceof Variable) {
			return "Variable: " + ((Variable)o).getURI(graph) + " " + NameUtils.getSafeLabel(graph, (Resource)((Variable)o).getPossiblePropertyValue(graph, Variables.TYPE));
		} else if(o instanceof SelectionInput) {
			return "SelectionInput: " + o.toString();
		} else {
			return "Object(" + o.getClass().getSimpleName() + "): " + o.toString();
		}
	}

}
