package org.simantics.scl.compiler.elaboration.expressions;

import org.simantics.scl.compiler.elaboration.contexts.TranslationContext;
import org.simantics.scl.compiler.elaboration.expressions.ERuleset.DatalogRule;
import org.simantics.scl.compiler.elaboration.expressions.block.RuleStatement;
import org.simantics.scl.compiler.elaboration.relations.LocalRelation;
import org.simantics.scl.compiler.errors.Locations;

import gnu.trove.map.hash.THashMap;

public class EPreRuleset extends ASTExpression {

    public RuleStatement[] statements;
    public Expression in;
    
    public EPreRuleset(RuleStatement[] statements, Expression in) {
        this.statements = statements;
        this.in = in;
    }

    @Override
    public Expression resolve(TranslationContext context) {
        THashMap<String, LocalRelation> relations = new THashMap<String, LocalRelation>(); 
        DatalogRule[] rules = new DatalogRule[statements.length];
        context.pushRelationFrame();
        try {
            for(int i=0;i<statements.length;++i) {
                RuleStatement statement = statements[i];
                Expression head = statement.head;
                if(!(head instanceof EApply)) {
                    context.getErrorLog().log(head.location, "Invalid rule head.");
                    return new EError();
                }
                EApply apply = (EApply)head;
                if(!(apply.function instanceof EVar)) {
                    context.getErrorLog().log(head.location, "Invalid relation in rule head.");
                    return new EError();
                }
                String relationName = ((EVar)apply.function).name;
                LocalRelation relation = relations.get(relationName);
                if(relation == null) {
                    relation = new LocalRelation(relationName, apply.parameters.length);
                    relations.put(relationName, relation);
                    context.newRelation(relationName, relation);
                }
                else if(apply.parameters.length != relation.getArity()) {
                    context.getErrorLog().log(apply.location, "Different rules have different relation arity.");
                    return new EError();
                }
                rules[i] = new DatalogRule(relation, apply.parameters, statement.body);
            }
            for(DatalogRule rule : rules) {
                context.pushExistentialFrame();
                for(int i=0;i<rule.headParameters.length;++i)
                    rule.headParameters[i] = rule.headParameters[i].resolve(context);
                rule.body = rule.body.resolve(context);
                rule.variables = context.popExistentialFrame();
            }
            return new ERuleset(
                    relations.values().toArray(new LocalRelation[relations.size()]),
                    rules,
                    in.resolve(context));
        } finally {
            context.popRelationFrame();
        }
    }

    @Override
    public void setLocationDeep(long loc) {
        if(location == Locations.NO_LOCATION) {
            location = loc;
            for(RuleStatement statement : statements)
                statement.setLocationDeep(loc);
            in.setLocationDeep(loc);
        }
    }
    
    @Override
    public Expression accept(ExpressionTransformer transformer) {
        return transformer.transform(this);
    }
    
    @Override
    public int getSyntacticFunctionArity() {
        return in.getSyntacticFunctionArity();
    }

}
