package org.simantics.scl.compiler.elaboration.expressions;

import org.simantics.scl.compiler.compilation.CompilationContext;
import org.simantics.scl.compiler.elaboration.contexts.ReplaceContext;
import org.simantics.scl.compiler.elaboration.contexts.SimplificationContext;
import org.simantics.scl.compiler.elaboration.contexts.TranslationContext;
import org.simantics.scl.compiler.elaboration.contexts.TypingContext;
import org.simantics.scl.compiler.errors.Locations;
import org.simantics.scl.compiler.internal.codegen.references.IVal;
import org.simantics.scl.compiler.internal.codegen.writer.CodeWriter;
import org.simantics.scl.compiler.internal.interpreted.IConstant;
import org.simantics.scl.compiler.internal.interpreted.IExpression;
import org.simantics.scl.compiler.internal.interpreted.IIf;
import org.simantics.scl.compiler.top.ExpressionInterpretationContext;
import org.simantics.scl.compiler.types.Type;
import org.simantics.scl.compiler.types.Types;
import org.simantics.scl.compiler.types.exceptions.MatchException;
import org.simantics.scl.runtime.tuple.Tuple0;

public class EIf extends Expression {
    public Expression condition;
    public Expression then_;
    public Expression else_; // may be null
    
    public EIf(Expression condition, Expression then_, Expression else_) {
        this.condition = condition;
        this.then_ = then_;
        this.else_ = else_;
    }

    public EIf(long loc, Expression condition, Expression then_, Expression else_) {
        super(loc);
        this.condition = condition;
        this.then_ = then_;
        this.else_ = else_;
    }

    @Override
    protected void updateType() throws MatchException {
        setType(then_.getType());
    }

    @Override
    public IVal toVal(CompilationContext context, CodeWriter w) {
        IVal conditionVal = condition.toVal(context, w); 
        CodeWriter joinPoint = w.createBlock(getType());
        CodeWriter thenBlock = w.createBlock();
        if(else_ != null) {
            CodeWriter elseBlock = w.createBlock();        
            w.if_(location, conditionVal, thenBlock.getContinuation(), elseBlock.getContinuation());
                
            IVal elseVal = else_.toVal(context, elseBlock);
            elseBlock.jump(location, joinPoint.getContinuation(), elseVal);
        }
        else {
            w.if_(location, conditionVal, thenBlock.getContinuation(), joinPoint.getContinuation());
        }
        IVal thenVal = then_.toVal(context, thenBlock);
        thenBlock.jump(location, joinPoint.getContinuation(), thenVal);
        w.continueAs(joinPoint);
        
        return w.getParameters()[0];
    }

    @Override
    public Expression simplify(SimplificationContext context) {
        condition = condition.simplify(context);
        then_ = then_.simplify(context);
        if(else_ != null)
            else_ = else_.simplify(context);
        return this;
    }

    @Override
    public Expression resolve(TranslationContext context) {
        condition = condition.resolve(context);
        then_ = then_.resolve(context);
        if(else_ != null)
            else_ = else_.resolve(context);
        return this;
    }

    @Override
    public Expression replace(ReplaceContext context) {
        return new EIf(condition.replace(context), 
                then_.replace(context), 
                else_ == null ? null : else_.replace(context));
    }
    
    @Override
    public Expression checkBasicType(TypingContext context, Type requiredType) {
        condition = condition.checkType(context, Types.BOOLEAN);
        then_ = then_.checkType(context, requiredType);
        if(else_ != null)
            else_ = else_.checkType(context, requiredType);
        else
            context.getErrorLog().log(location, "Else branch is required because the return value of the if expression is used.");
        return this;
    }
    
    @Override
    public Expression checkIgnoredType(TypingContext context) {
        condition = condition.checkType(context, Types.BOOLEAN);
        then_ = then_.checkIgnoredType(context);
        if(else_ != null)
            else_ = else_.checkIgnoredType(context);
        return this;
    }
    
    @Override
    public boolean isEffectful() {
    	return condition.isEffectful() || then_.isEffectful() || (else_ != null && else_.isEffectful());
    }
    
    @Override
    public void setLocationDeep(long loc) {
        if(location == Locations.NO_LOCATION) {
            location = loc;
            condition.setLocationDeep(loc);
            then_.setLocationDeep(loc);
            if(else_ != null)
                else_.setLocationDeep(loc);
        }
    }
    
    @Override
    public void accept(ExpressionVisitor visitor) {
        visitor.visit(this);
    }
    
    @Override
    public IExpression toIExpression(ExpressionInterpretationContext target) {
        return new IIf(condition.toIExpression(target), then_.toIExpression(target), 
                else_ != null ? else_.toIExpression(target) : new IConstant(Tuple0.INSTANCE));
    }

    @Override
    public Expression accept(ExpressionTransformer transformer) {
        return transformer.transform(this);
    }

    @Override
    public int getSyntacticFunctionArity() {
        return Math.max(then_.getSyntacticFunctionArity(), else_.getSyntacticFunctionArity());
    }
}
