/*******************************************************************************
 * Copyright (c) 2007, 2013 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.objmap.structural.schema;



import java.util.Stack;

import gnu.trove.map.hash.THashMap;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.objmap.exceptions.MappingException;
import org.simantics.objmap.graph.schema.ILinkType;
import org.simantics.objmap.graph.schema.IMappingSchema;
import org.simantics.objmap.structural.IStructuralObject;
import org.simantics.objmap.structural.StructuralResource;


/**
 * 
 */
public class DefaultSchema implements IMappingSchema<StructuralResource, IStructuralObject> {

    THashMap<Resource, ILinkType<StructuralResource, IStructuralObject>> domainLinkTypes = 
        new THashMap<Resource, ILinkType<StructuralResource, IStructuralObject>>();
    THashMap<Class<?>, ILinkType<StructuralResource, IStructuralObject>> rangeLinkTypes = 
        new THashMap<Class<?>, ILinkType<StructuralResource, IStructuralObject>>();
    
    public void addLinkType(SimpleLinkType linkType) {
        domainLinkTypes.put(linkType.domainType, linkType);
        rangeLinkTypes.put(linkType.rangeType, linkType);
    }
    
    public void addLinkType(AdaptedLinkType linkType) {
        domainLinkTypes.put(linkType.domainType, linkType);
        rangeLinkTypes.put(linkType.rangeType, linkType);
    }
    
    @Override
    public ILinkType<StructuralResource, IStructuralObject> linkTypeOfDomainElement(ReadGraph g, StructuralResource element) throws MappingException {        
        try {
        	
        	for(Resource type : g.getTypes(element.getResource())) {

        		ILinkType<StructuralResource, IStructuralObject> linkType = domainLinkTypes.get(type);
        		if(linkType != null) return linkType;
        		
        	}
        	
        	throw new MappingException("Didn't find a link type for " +
        			NameUtils.getSafeName(g, element.getResource()) + ".");
   				
        } catch (DatabaseException e) {
            throw new MappingException(e);
        }
    }

    @Override
    public ILinkType<StructuralResource, IStructuralObject> linkTypeOfRangeElement(IStructuralObject element) throws MappingException {
    	ILinkType<StructuralResource, IStructuralObject> type = rangeLinkTypes.get(element.getClass());
		if(type == null)  {
			Stack<Class<?>> clazzes = new Stack<Class<?>>();
    		for (Class<?> clazz : element.getClass().getInterfaces()) {
    			clazzes.add(clazz);
    		}
    		clazzes.add(element.getClass().getSuperclass());
    		
			while (!clazzes.isEmpty()) {
				Class<?> clazz = clazzes.pop();
			
				type = rangeLinkTypes.get(clazz);
				if (type != null)
					return type;
				for (Class<?> c : clazz.getInterfaces())
					clazzes.add(c);
				
			}
			throw new MappingException("Didn't find a link type for " +	element + ".");
		}
		return type;
    }

    
    public ILinkType<StructuralResource, IStructuralObject> linkTypeOfDomainType(ReadGraph g, Resource type)  {        
    	return domainLinkTypes.get(type);
    }
    
    public ILinkType<StructuralResource, IStructuralObject> linkTypeOfRangeType(Class<?> clazz) {
    	ILinkType<StructuralResource, IStructuralObject> type = rangeLinkTypes.get(clazz);
    	if(type == null)  {
    		Stack<Class<?>> clazzes = new Stack<Class<?>>();
    		for (Class<?> c : clazz.getInterfaces()) {
    			clazzes.add(c);
    		}
    		clazzes.add(clazz.getSuperclass());
    		
    		while (!clazzes.isEmpty()) {
				Class<?> c = clazzes.pop();
			
				type = rangeLinkTypes.get(c);
				if (type != null)
					return type;
				for (Class<?> c2 : c.getInterfaces())
					clazzes.add(c2);
				
			}
    		
		}
    	return null;
    }
}
