/*******************************************************************************
 * Copyright (c) 2007, 2013 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.objmap.structural.rules.domain;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.objmap.exceptions.MappingException;
import org.simantics.objmap.graph.rules.domain.IDomainAccessor;
import org.simantics.objmap.structural.StructuralResource;


public class RelatedObjectAccessor implements IDomainAccessor<StructuralResource,StructuralResource> {

    static Logger LOGGER = LoggerFactory.getLogger(RelatedObjectAccessor.class);
    
	Resource relation;
	boolean useTypeResource;
	
	private boolean preventStructuralChanges = true;
	private boolean preventStructuralRootChanges = true;
	
	public RelatedObjectAccessor(Resource relation, boolean useTypeResource) {
		this.relation = relation;
		this.useTypeResource = useTypeResource;
	}
	
	public RelatedObjectAccessor(Resource relation, boolean useTypeResource, boolean preventStructuralChanges, boolean preventStructuralRootChanges) {
		this.relation = relation;
		this.useTypeResource = useTypeResource;
		this.preventStructuralChanges = preventStructuralChanges;
		this.preventStructuralRootChanges = preventStructuralRootChanges;
	}
	
 private boolean preventChange(StructuralResource element) {
    	return preventStructuralChanges && element.isStructural() && (!element.isStructuralRoot()||preventStructuralRootChanges);	
    }

	@Override
	public StructuralResource get(ReadGraph g, StructuralResource element) throws MappingException {
		try {
		    LOGGER.trace("        RelatedObjectAccessor.get");
		    Resource res = getServiceResource(g, element);
		    if (res == null)
		    	return null;
			Resource r =  g.getPossibleObject(res, relation);
			if (r == null)
				return null;
			if (StructuralUtils.isStructuralInstance(g, r)) {
				return new StructuralResource(g, r, element.getContext(),r);
			} else {
				return new StructuralResource(g, r, element.getContext());
			}
		} catch (DatabaseException e) {
			throw new MappingException(e);
		}
	}
	
	@Override
	public boolean set(WriteGraph g, StructuralResource selement, StructuralResource value)
			throws MappingException {
		try {
		    LOGGER.trace("        RelatedObjectAccessor.set");
		    Resource element = getServiceResource(g, selement);
		    if (element == null)
		    	return false;
		    Resource resource = g.getPossibleObject(element, relation);
			if(resource == null) {
			    if(value == null)
			        return false;
			    if (preventChange(selement))
					return false;
			    g.claim(element, relation, value.getResource());
			    return true;
			}
			else if(resource.equals(value.getResource()))
			    return false;
			else {
				if (preventChange(selement))
					return false;
			    g.deny(element, relation);
			    if(value != null)
			        g.claim(element, relation, value.getResource());
			    return true;
			}
		} catch (DatabaseException e) {
			throw new MappingException(e);
		}
		
	}
	
		
	
	private Resource getServiceResource(ReadGraph g, StructuralResource element) {
		if(!useTypeResource)
			return element.getResource();
		return element.getTypeResource();
	}

}
