/*******************************************************************************
 * Copyright (c) 2012 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.sharedontology.wizard;

import java.io.File;
import java.io.IOException;
import java.util.Collections;
import java.util.List;

import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.simantics.databoard.Bindings;
import org.simantics.databoard.adapter.AdaptException;
import org.simantics.databoard.binding.mutable.Variant;
import org.simantics.databoard.container.DataContainer;
import org.simantics.databoard.container.DataContainers;
import org.simantics.db.common.NamedResource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.util.DraftStatusBean;
import org.simantics.utils.ui.ErrorLogger;

/**
 * @author Tuukka Lehtonen
 */
public class SharedOntologyImportPage extends WizardPage {

    /**
     * If non-null, the wizard cannot continue. This message tells why.
     */
    String              failure;

    ImportPlan          importModel;

    Composite           draft;
    
    Text                importTarget;
    CCombo              importLocation;

    List<NamedResource> models = Collections.emptyList();
    Label               author;
    Label               status;

    protected SharedOntologyImportPage(ImportPlan model) {
        super("Import Shared Library", "Define Import Location", null);
        this.importModel = model;
    }

    @Override
    public void createControl(Composite parent) {
        Composite container = new Composite(parent, SWT.NONE);
        {
            GridLayout layout = new GridLayout();
            layout.horizontalSpacing = 20;
            layout.verticalSpacing = 10;
            layout.numColumns = 3;
            container.setLayout(layout);
        }

        draft = new Composite(container, SWT.NONE);
        draft.setBackground(draft.getDisplay().getSystemColor(SWT.COLOR_RED));
        GridDataFactory.fillDefaults().grab(true, false).span(3, 1).applyTo(draft);
    	GridLayoutFactory.swtDefaults().numColumns(0).margins(0, 0).applyTo(draft);
        
        Composite draft2 = new Composite(draft, SWT.NONE);
        GridLayoutFactory.swtDefaults().applyTo(draft2);
        GridDataFactory.fillDefaults().grab(true, false).span(1, 1).applyTo(draft2);
        new Label(draft2, SWT.NONE).setText("This library draft was not finished for publishing.");
        
        new Label(container, SWT.NONE).setText("&Shared library file:");
        importLocation = new CCombo(container, SWT.BORDER);
        {
            importLocation.setText("");
            GridDataFactory.fillDefaults().grab(true, false).span(1, 1).applyTo(importLocation);
            importLocation.addModifyListener(new ModifyListener(){
                @Override
                public void modifyText(ModifyEvent e) {
                    validatePage();
                }
            });
        }
        Button browseFileButton = new Button(container, SWT.PUSH);
        {
            browseFileButton.setText("Br&owse...");
            browseFileButton.setLayoutData(new GridData(SWT.RIGHT, SWT.CENTER, false, false));
            browseFileButton.addSelectionListener(new SelectionAdapter() {
                @Override
                public void widgetSelected(SelectionEvent e) {
                    FileDialog dialog = new FileDialog(getShell(), SWT.OPEN);
                    dialog.setText("Choose Shared Library to Import");
                    String loc = importLocation.getText();
                    dialog.setFilterPath(loc);
                    dialog.setFilterExtensions(new String[] { "*.sharedLibrary" });
                    dialog.setFilterNames(new String[] { "Shared Library (*.sharedLibrary)" });
                    String file = dialog.open();
                    if (file == null)
                        return;
                    importLocation.setText(file);
                    validatePage();
                }
            });
        }
        
        author = new Label(container, SWT.NONE);
        author.setText("");
        GridDataFactory.fillDefaults().grab(true, false).span(3, 1).applyTo(author);

        status = new Label(container, SWT.NONE);
        status.setText("");
        GridDataFactory.fillDefaults().grab(true, false).span(3, 1).applyTo(status);

        try {
            initializeData();
        } catch (DatabaseException e) {
            ErrorLogger.defaultLogError(e);
        }

        setControl(container);
        validatePage();
    }

    private void initializeData() throws DatabaseException {
        for (String path : importModel.recentLocations) {
            importLocation.add(path);
        }
        if (importLocation.getItemCount() > 0)
            importLocation.select(0);
    }

    void validatePage() {
    	
        if (failure != null) {
            setErrorMessage(failure);
            setPageComplete(false);
            return;
        }
        String importLoc = importLocation.getText();
        if (importLoc.isEmpty()) {
            setMessage("Select file to import.");
            setErrorMessage(null);
            setPageComplete(false);
            return;
        }
        File file = new File(importLoc);
        if (!file.exists() || !file.isFile()) {
            setErrorMessage("Selected file is invalid.");
            setPageComplete(false);
            return;
        }
        importModel.importLocation = file;

        try {

        	DataContainer container = DataContainers.readHeader(file);
        	Variant draftStatus = container.metadata.get(DraftStatusBean.EXTENSION_KEY);
        	if(draftStatus != null) {
        		GridLayoutFactory.swtDefaults().spacing(5, 5).applyTo(draft);
        		draft.getParent().layout(true);
        	} else {
        		GridLayoutFactory.swtDefaults().numColumns(0).margins(0, 0).applyTo(draft);
        		draft.getParent().layout(true);
        	}
        	
        	Variant authorVariant = container.metadata.get("author");
        	Variant dateVariant = container.metadata.get("date");
        	
        	if(authorVariant != null && dateVariant != null) {
        		String auth = (String)authorVariant.getValue(Bindings.STRING);
        		String date = (String)dateVariant.getValue(Bindings.STRING);
        		author.setText("Created by " + auth + " on " + date);
        	} else {
        		author.setText("");
        	}

        } catch (IOException e) {
        	setErrorMessage("Could not read header information from " + file.getAbsolutePath());
        	setPageComplete(false);
        	return;
        } catch (AdaptException e) {
        	setErrorMessage("Could not read header information from " + file.getAbsolutePath());
        	setPageComplete(false);
        	return;
		}           
        
        setErrorMessage(null);
        setMessage("Ready to import " + file.getName() + "");
        setPageComplete(true);
        
    }

}
