/*******************************************************************************
 * Copyright (c) 2017 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.sharedontology.wizard;

import java.io.File;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubMonitor;
import org.simantics.databoard.adapter.AdaptException;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.container.DataContainer;
import org.simantics.databoard.container.DataContainers;
import org.simantics.databoard.container.FormatHandler;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.common.primitiverequest.IsExternalEntity;
import org.simantics.db.common.primitiverequest.PossibleResource;
import org.simantics.db.layer0.adapter.impl.DefaultPasteImportAdvisor;
import org.simantics.db.layer0.migration.MigratedImportResult;
import org.simantics.db.layer0.migration.MigrationState;
import org.simantics.db.layer0.migration.MigrationStateKeys;
import org.simantics.db.layer0.migration.MigrationUtils;
import org.simantics.db.layer0.util.ModelDependenciesBean;
import org.simantics.db.layer0.util.ModelDependency;
import org.simantics.graph.db.ImportResult;
import org.simantics.graph.representation.TransferableGraph1;
import org.simantics.modeling.ui.Activator;

/**
 * @author Tuukka Lehtonen
 * @since 1.31.0
 */
public class ModelImporter {

    public static MigratedImportResult doImport(IProgressMonitor monitor, File modelFile, Session session, Resource target, boolean includeDependencies)
            throws Exception
    {
        SubMonitor mon = SubMonitor.convert(monitor);
        mon.beginTask("Loading model from disk", 1000);

        FormatHandler<MigratedImportResult> handler1 = new FormatHandler<MigratedImportResult>() {
            @Override
            public Binding getBinding() {
                return TransferableGraph1.BINDING;
            }

            @Override
            public MigratedImportResult process(DataContainer container) throws Exception {
                mon.worked(100);
                mon.setTaskName("Importing model into database");

                MigrationState state = MigrationUtils.newState();
                state.setProperty(MigrationStateKeys.UPDATE_DEPENDENCIES, false);
                state.setProperty(MigrationStateKeys.MODEL_FILE, modelFile);
                state.setProperty(MigrationStateKeys.SESSION, session);
                state.setProperty(MigrationStateKeys.PROGRESS_MONITOR, monitor);
                try {
	                if (includeDependencies) {
	                    try {
	                        ModelDependenciesBean libraryDependenciesBean = ModelDependenciesBean.fromMigrationState(state);
	                        if (libraryDependenciesBean != null) {
	                            for (ModelDependency dependency : libraryDependenciesBean.dependencies) {
	                                Resource existing = session.sync(new PossibleResource(dependency.uri));
	                                boolean isExternalEntity = existing != null && session.syncRequest(new IsExternalEntity(existing));
	                                if (existing == null || isExternalEntity) {
	                                    MigrationUtils.importSharedOntology(session, dependency.tg, false);
	                                }
	                            }
	                        }
	                    } catch (AdaptException e) {
	                        Activator.getDefault().getLog().log(new Status(IStatus.ERROR, Activator.PLUGIN_ID, "Could not read model dependencies bean.", e));
	                    }
	                }
	
	                MigrationUtils.importMigrated(monitor, session, modelFile, state, new DefaultPasteImportAdvisor(target), target);
	
	                Collection<Resource> resultRoots = state.getProperty(MigrationStateKeys.CURRENT_ROOT_RESOURCES);
	                ImportResult result = state.getProperty(MigrationStateKeys.IMPORT_RESULT);
	                return new MigratedImportResult(resultRoots, result);
                } finally {
                   state.dispose();
                }
            }
        };

        Map<String, FormatHandler<MigratedImportResult>> handlers = new HashMap<>();
        handlers.put(":1", handler1);
        handlers.put(Constants.MODEL_FORMAT_V1, handler1);

        MigratedImportResult result = DataContainers.readFile(modelFile, handlers);

        mon.setTaskName("Postprocessing");
        mon.subTask("");
        mon.newChild(50).done();

        return result;
    }

}
