/*******************************************************************************
 * Copyright (c) 2012 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.sharedontology.wizard;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.SubMonitor;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.simantics.Simantics;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.mutable.Variant;
import org.simantics.databoard.serialization.SerializationException;
import org.simantics.db.ReadGraph;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.common.utils.Logger;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.util.ModelDependenciesBean;
import org.simantics.db.layer0.util.TransferableGraphConfiguration2;
import org.simantics.modeling.ModelingUtils;
import org.simantics.modeling.ModelingUtils.LibraryInfo;
import org.simantics.utils.ui.dialogs.ShowMessage;

/**
 * @author Antti Villberg
 */
public class ModelExporter implements IRunnableWithProgress {

    ExportPlan exportModel;

    public ModelExporter(ExportPlan exportModel) {
        this.exportModel = exportModel;
    }

    @Override
    public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
        SubMonitor progress = SubMonitor.convert(monitor, 50);
        try {
            exportModel(progress.newChild(50, SubMonitor.SUPPRESS_NONE));
        } catch (IOException e) {
            throw new InvocationTargetException(e);
        } catch (DatabaseException e) {
            throw new InvocationTargetException(e);
        } catch (BindingException e) {
            throw new InvocationTargetException(e);
        } finally {
            monitor.done();
        }
    }

    void exportModel(SubMonitor mon) throws IOException, DatabaseException, SerializationException, BindingException{
        try {
            doExport(mon, exportModel.exportLocation, exportModel.model, exportModel.includeDependencies);
        } catch (DatabaseException e) {
            e.printStackTrace();
            Logger.defaultLogError(e);
            mon.setCanceled(true);
            ShowMessage.showError("Export failed.", "Internal application error in export. See log for details.");
        } finally {
            mon.setWorkRemaining(0);
        }
    }

    public static void doExport(IProgressMonitor monitor, final File location, final LibraryInfo info, boolean includeDependencies) throws DatabaseException, IOException {
        Simantics.sync(new ReadRequest() {
            @Override
            public void run(ReadGraph graph) throws DatabaseException {
                TransferableGraphConfiguration2 conf = new TransferableGraphConfiguration2(graph, info.library.getResource(), true, false);
                if (includeDependencies) {
                    ModelDependenciesBean bean = ModelDependenciesBean.create(graph, info.library.getResource());
                    conf.baseExtensions.put(ModelDependenciesBean.EXTENSION_KEY, new Variant(ModelDependenciesBean.BINDING, bean));
                }
                ModelingUtils.exportModel(graph, conf, location.getAbsolutePath(), "", 1);
            }
        });
    }

}
