/*******************************************************************************
 * Copyright (c) 2017 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.scl.scriptEditor;

import org.eclipse.core.commands.AbstractHandler;
import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.text.IDocument;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.handlers.HandlerUtil;
import org.eclipse.ui.texteditor.ITextEditor;
import org.simantics.Simantics;
import org.simantics.db.RequestProcessor;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.common.primitiverequest.PossibleResource;
import org.simantics.db.common.uri.ResourceToPossibleURI;
import org.simantics.db.exception.DatabaseException;
import org.simantics.modeling.scl.SCLScripts;
import org.simantics.modeling.ui.Activator;
import org.simantics.scl.compiler.commands.CommandSession;
import org.simantics.scl.runtime.reporting.SCLReportingHandler;
import org.simantics.utils.datastructures.Pair;
import org.simantics.utils.ui.workbench.WorkbenchUtils;

/**
 * @author Tuukka Lehtonen
 * @since 1.31.0
 */
public class RunSCLScriptHandler extends AbstractHandler {

	@Override
	public Object execute(ExecutionEvent event) throws ExecutionException {
		IEditorPart editor = HandlerUtil.getActiveEditorChecked(event);
		IStatusLineManager status = WorkbenchUtils.getStatusLine(editor);
		try {
			Session s = Simantics.getSession();
			final Resource script = getInputResource(s, editor);
			if (script == null)
				return null;
			String error = SCLScripts.canRunScript(s, script);
			if (error == null) {
				String textSnapshot = getDocumentText(editor);
				if (textSnapshot != null) {
					String scriptUrl = s.syncRequest(new ResourceToPossibleURI(script));
					Pair<CommandSession, SCLReportingHandler> p = SCLScripts.getOrCreateConsoleCommandSession();
					p.first.setRelativeResolutionModuleName(scriptUrl);
					SCLScripts.runScriptWithProgress(editor.getTitle(), textSnapshot, p.first, p.second);
					status.setErrorMessage(null);
				}
			} else {
				status.setErrorMessage(error);
			}
		} catch (DatabaseException ex) {
			Activator.getDefault().getLog().log(new Status(IStatus.ERROR, Activator.PLUGIN_ID, "Failed to run SCL script.", ex));
		}
		return null;
	}

	private static Resource getInputResource(RequestProcessor processor, IEditorPart editor) throws DatabaseException {
		IEditorInput input = editor.getEditorInput();
		return (input instanceof SCLScriptEditorInput)
				? processor.syncRequest(new PossibleResource(((SCLScriptEditorInput) input).getScriptURI()))
				: null;
	}

	private static String getDocumentText(IEditorPart editor) {
		if (!(editor instanceof ITextEditor))
			return null;
		IDocument document = ((ITextEditor) editor).getDocumentProvider().getDocument(editor.getEditorInput());
		return document == null ? null : document.get();
	}

}
