/*******************************************************************************
 * Copyright (c) 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.property.svg;

import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.Set;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.simantics.browsing.ui.swt.widgets.impl.ReadFactoryImpl;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.diagram.stubs.G2DResource;
import org.simantics.image2.ontology.ImageResource;
import org.w3c.dom.Document;
import org.w3c.dom.bootstrap.DOMImplementationRegistry;
import org.w3c.dom.ls.DOMImplementationLS;
import org.w3c.dom.ls.LSOutput;
import org.w3c.dom.ls.LSSerializer;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;


/**
 * @author Antti Villberg
 */
public class SVGTextFactory extends ReadFactoryImpl<SVGInput, String> {

    @Override
    public String perform(ReadGraph graph, SVGInput input) throws DatabaseException {
        Resource res = input.getDatum();
        String possibleText = possibleSVGText(graph, res);
        if(possibleText != null) {
            return XMLPrettyPrinter.safePrettyPrint(possibleText);
        } else {
            return "";
        }
    }
    
    private String possibleSVGText(ReadGraph graph, Resource res) throws DatabaseException {
        G2DResource G2D = G2DResource.getInstance(graph);
        DiagramResource DIA = DiagramResource.getInstance(graph);
        ImageResource IMG = ImageResource.getInstance(graph);
        Set<Resource> types = graph.getTypes(res);
        if(types.contains(DIA.SVGElement)) {
            return graph.getPossibleRelatedValue(res, G2D.HasSVGDocument, Bindings.STRING);
        } else if(types.contains(IMG.SvgImage)) {
            return graph.getPossibleValue(res, Bindings.STRING);
        } else {
            return null;
        }
    }
    

}

/**
 * @author Antti Villberg
 */
class XMLPrettyPrinter {

    private static DOMImplementationRegistry registry;

    static {
        try {
            registry = DOMImplementationRegistry.newInstance();
        } catch (ClassCastException e) {
            e.printStackTrace();
        } catch (ClassNotFoundException e) {
            e.printStackTrace();
        } catch (InstantiationException e) {
            e.printStackTrace();
        } catch (IllegalAccessException e) {
            e.printStackTrace();
        }
    }

    public static Document parseDocument(String document) throws IOException, SAXException, ParserConfigurationException {
        InputSource xmlSource = new InputSource(new StringReader(document)); 
        DocumentBuilderFactory dbFactory = DocumentBuilderFactory.newInstance();
        dbFactory.setExpandEntityReferences(false);
        dbFactory.setValidating(false);
//        dbFactory.setFeature("http://xml.org/sax/features/validation", false);
//        dbFactory.setFeature("http://xml.org/sax/features/external-general-entities", false);
        dbFactory.setFeature("http://xml.org/sax/features/external-parameter-entities", false);
//        dbFactory.setFeature("http://xml.org/sax/features/namespaces", false);
//        dbFactory.setFeature("http://apache.org/xml/features/validation/schema", false);
//        dbFactory.setFeature("http://apache.org/xml/features/validation/schema-full-checking", false);
        DocumentBuilder dBuilder = dbFactory.newDocumentBuilder();
        return dBuilder.parse(xmlSource);
    }

    public static String prettyPrint(String document) throws IOException, SAXException, ParserConfigurationException {
//        long start = System.nanoTime();
        Document doc = parseDocument(document);

        DOMImplementationLS impl = (DOMImplementationLS)registry.getDOMImplementation("LS");
        LSSerializer serializer = impl.createLSSerializer();
        serializer.getDomConfig().setParameter("format-pretty-print", Boolean.TRUE);
        LSOutput lsOutput = impl.createLSOutput();
        lsOutput.setEncoding("UTF-8");
        StringWriter stringWriter = new StringWriter();
        lsOutput.setCharacterStream(stringWriter);
        serializer.setNewLine("\n");
        serializer.write(doc, lsOutput);
        String formatted = stringWriter.toString();

//        long duration = System.nanoTime() - start;
//        System.out.println("Pretty print took " + 1e-9*duration + "s.");
//        System.out.println("original: " + document);
//        System.out.println("formatted: " + formatted);

        return formatted;
    }

    public static String safePrettyPrint(String document) {
        try {
            return prettyPrint(document);
        } catch (Exception e) {
            e.printStackTrace();
            return "";
        }
    }

}