/*******************************************************************************
 * Copyright (c) 2007, 2011, 2014 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.jfreechart.chart.properties.bar;

import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.ui.IWorkbenchSite;
import org.simantics.browsing.ui.swt.SingleSelectionInputSource;
import org.simantics.browsing.ui.swt.widgets.Button;
import org.simantics.browsing.ui.swt.widgets.GraphExplorerComposite;
import org.simantics.browsing.ui.swt.widgets.impl.SelectionListenerImpl;
import org.simantics.browsing.ui.swt.widgets.impl.Widget;
import org.simantics.browsing.ui.swt.widgets.impl.WidgetSupport;
import org.simantics.browsing.ui.swt.widgets.impl.WidgetSupportImpl;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.PossibleObjectWithType;
import org.simantics.db.common.utils.ListUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.util.RemoverUtil;
import org.simantics.db.management.ISessionContext;
import org.simantics.jfreechart.chart.ChartUtils;
import org.simantics.jfreechart.chart.properties.AdjustableTab;
import org.simantics.jfreechart.chart.properties.xyline.AxisAndVariablesExplorerComposite;
import org.simantics.layer0.Layer0;
import org.simantics.sysdyn.JFreeChartResource;
import org.simantics.utils.datastructures.ArrayMap;
import org.simantics.utils.ui.AdaptionUtils;

/**
 * Tab containing the series of a bar chart
 * @author Teemu Lempinen
 * @author Tuomas Miettinen
 *
 */
public class BarSeriesTab extends AdjustableTab implements Widget {

    private GraphExplorerComposite explorer;
    private ScrolledComposite propertyContainer;
    private WidgetSupportImpl additionalSupport;
    private Button add, remove;
    private Resource chartResource;
    private BarSeriesPropertyComposite spc;
    private int options;
	private Composite buttonComposite;

    public BarSeriesTab(Object id, int options) {
        super(id);
        additionalSupport = new WidgetSupportImpl();
        this.options = options;
    }

    /**
     * Updates the content of propertyContainer  
     * @param context
     */
    private void updateSelection(ISessionContext context) {
        ISelectionProvider selectionProvider = (ISelectionProvider)explorer.getAdapter(ISelectionProvider.class);
        IStructuredSelection selection = (IStructuredSelection)selectionProvider.getSelection();
        final Resource resource = AdaptionUtils.adaptToSingle(selection, Resource.class);
        if(resource == null)
            return;

        for(Control child : propertyContainer.getChildren()) {
            child.dispose();
        }
        spc = new BarSeriesPropertyComposite(propertyContainer, context, additionalSupport, options,SWT.NONE);

        additionalSupport.fireInput(context, selection);

        propertyContainer.setContent(spc);
        Point size = spc.computeSize(SWT.DEFAULT, SWT.DEFAULT);
        propertyContainer.setMinSize(size);
    }

    /**
     * SelectionListener for adding a new variable to a plot
     * @author Teemu Lempinen
     *
     */
    private class NewVariableListener extends SelectionListenerImpl<Resource> {

        public NewVariableListener(ISessionContext context) {
            super(context);
        }

        @Override
        public void apply(WriteGraph graph, Resource input) throws DatabaseException {
            Resource dataset = null;
            JFreeChartResource jfree = JFreeChartResource.getInstance(graph);
            Layer0 l0 = Layer0.getInstance(graph);

            if(input == null) {
                if(chartResource != null) {
                    Resource plot = graph.syncRequest(new PossibleObjectWithType(chartResource, l0.ConsistsOf, jfree.Plot));
                    if(plot != null)
                        dataset = graph.syncRequest(new PossibleObjectWithType(plot, l0.ConsistsOf, jfree.Dataset));
                }
            } else {
                if(graph.isInstanceOf(input, jfree.Series)) {
                    dataset = graph.getPossibleObject(input, l0.PartOf);
                }
            }

            if(dataset != null) {
                // Create series with no rvi
                ChartUtils.createSeries(graph, dataset, null);
            }
        }
    }

    /**
     * SelectionListener for remove button
     * @author Teemu Lempinen
     *
     */
    private class RemoveListener extends SelectionListenerImpl<Resource> {

        public RemoveListener(ISessionContext context) {
            super(context);
        }

        /**
         * Removes selected resource from explorer
         */
        @Override
        public void apply(WriteGraph graph, Resource input) throws DatabaseException {
            if(input == null)
                return; 

            JFreeChartResource jfree = JFreeChartResource.getInstance(graph);
            Layer0 l0 = Layer0.getInstance(graph);
            Resource list = null;
            if(graph.isInstanceOf(input, jfree.Series)) {
                // Remove series from dataset and seriesList
                Resource dataset = graph.getPossibleObject(input, l0.PartOf);
                if(dataset != null)
                    list = graph.getPossibleObject(dataset, jfree.Dataset_seriesList);

                if(list != null)
                    ListUtils.removeElement(graph, list, input);
                RemoverUtil.remove(graph, input);
            }
        }
    }

    @Override
    public void setInput(ISessionContext context, Object input) {
        chartResource = AdaptionUtils.adaptToSingle(input, Resource.class);
    }

	@Override
	protected void createAndAddControls(Composite body, IWorkbenchSite site,
			final ISessionContext context, WidgetSupport support) {
		support.register(this);
        composite = new Composite(body, SWT.NONE);

        // (Ontology-based) GraphExplorer displaying variables in a bar chart
        explorer = new AxisAndVariablesExplorerComposite(ArrayMap.keys(
                "displaySelectors", "displayFilter").values(false, false), site, composite, support, SWT.FULL_SELECTION | SWT.BORDER | SWT.SINGLE);
        explorer.setBrowseContexts(JFreeChartResource.URIs.BarSeriesBrowseContext);
        explorer.setInputSource(new SingleSelectionInputSource(
                Resource.class));
        explorer.getExplorer().setAutoExpandLevel(2); // Expand everything in the beginning
        explorer.finish();

        ((Tree)explorer.getExplorerControl()).addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
                updateSelection(context);
            }
        });

        // Scrolled composite for displaying properties of a selection in explorer
        propertyContainer = new ScrolledComposite(composite, SWT.H_SCROLL | SWT.V_SCROLL);
        propertyContainer.setExpandHorizontal(true);
        propertyContainer.setExpandVertical(true);

        // Buttons for adding and removing variables from a pie plot
        buttonComposite = new Composite(composite, SWT.NONE);

        add = new Button(buttonComposite, additionalSupport, SWT.NONE);
        add.setText("Add");
        add.addSelectionListener(new NewVariableListener(context));

        remove = new Button(buttonComposite, additionalSupport, SWT.NONE);
        remove.setText("Remove");
        remove.addSelectionListener(new RemoveListener(context));
	}

	@Override
	protected void createControlLayoutVertical() {
        GridDataFactory.fillDefaults().grab(true, true).applyTo(composite);
        GridLayoutFactory.fillDefaults().numColumns(1).margins(3, 3).applyTo(composite);

        GridDataFactory.fillDefaults().hint(150, 200).grab(false, true).applyTo(explorer);

        // Scrolled composite for displaying properties of a selection in explorer
        GridDataFactory.fillDefaults().hint(150, 130).span(1, 1).grab(true, false).applyTo(propertyContainer);
        GridLayoutFactory.fillDefaults().applyTo(propertyContainer);

        // Buttons for adding and removing variables from a pie plot
        GridDataFactory.fillDefaults().applyTo(buttonComposite);
        GridLayoutFactory.fillDefaults().numColumns(3).applyTo(buttonComposite);
	}

	@Override
	protected void createControlLayoutHorizontal(boolean wideScreen) {
        GridDataFactory.fillDefaults().grab(true, true).applyTo(composite);
        GridLayoutFactory.fillDefaults().numColumns(2).margins(3, 3).applyTo(composite);

        GridDataFactory.fillDefaults().hint(250, SWT.DEFAULT).grab(false, true).applyTo(explorer);

        // Scrolled composite for displaying properties of a selection in explorer
        GridDataFactory.fillDefaults().span(1, 2).grab(true, true).applyTo(propertyContainer);
        GridLayoutFactory.fillDefaults().applyTo(propertyContainer);

        // Buttons for adding and removing variables from a pie plot
        GridDataFactory.fillDefaults().applyTo(buttonComposite);
        GridLayoutFactory.fillDefaults().numColumns(3).applyTo(buttonComposite);
	}


}
