/*******************************************************************************
 * Copyright (c) 2007, 2011, 2014 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.jfreechart.chart.properties.bar;

import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Spinner;
import org.eclipse.ui.IWorkbenchSite;
import org.simantics.browsing.ui.swt.widgets.StringPropertyFactory;
import org.simantics.browsing.ui.swt.widgets.StringPropertyModifier;
import org.simantics.browsing.ui.swt.widgets.TrackedText;
import org.simantics.browsing.ui.swt.widgets.impl.ReadFactoryImpl;
import org.simantics.browsing.ui.swt.widgets.impl.TextModifyListener;
import org.simantics.browsing.ui.swt.widgets.impl.TrackedModifyEvent;
import org.simantics.browsing.ui.swt.widgets.impl.Widget;
import org.simantics.browsing.ui.swt.widgets.impl.WidgetSupport;
import org.simantics.browsing.ui.swt.widgets.impl.WidgetSupportImpl;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.PossibleObjectWithType;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.management.ISessionContext;
import org.simantics.jfreechart.chart.properties.AdjustableTab;
import org.simantics.jfreechart.chart.properties.AxisHidePropertyComposite;
import org.simantics.jfreechart.chart.properties.ColorPicker;
import org.simantics.jfreechart.chart.properties.DoubleValidator;
import org.simantics.jfreechart.chart.properties.JFreeChartPropertyColorProvider;
import org.simantics.jfreechart.chart.properties.TrackedSpinner;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ui.chart.property.DoublePropertyFactory;
import org.simantics.modeling.ui.chart.property.DoublePropertyModifier;
import org.simantics.sysdyn.JFreeChartResource;
import org.simantics.utils.ui.AdaptionUtils;

/**
 * Tab for bar chart axis properties
 * @author Teemu Lempinen
 * @author Tuomas Miettinen
 *
 */
public class BarAxisTab extends AdjustableTab implements Widget {

    public BarAxisTab(Object id) {
        super(id);
    }

    private TrackedSpinner angle;
    private Integer angleInt = null;
    private WidgetSupportImpl domainAxisSupport = new WidgetSupportImpl();
    private WidgetSupportImpl rangeAxisSupport = new WidgetSupportImpl();
    private TrackedText rangelabel, rangemin, rangemax;
    private ScrolledComposite sc;
	private Group domainGroup;
	private Label labelLabel;
	private AxisHidePropertyComposite axisHide;
	private Label angleLabel;
	private Composite angleComposite;
	private Label labelColor;
	private ColorPicker colorPicker;
	private Group rangeGroup;
	private Label labelLabel2;
	private Label labelMin;
	private Composite minmax;
	private Label labelMax;
	private Label labelColor2;
	private TrackedText rangelabel2;
	private AxisHidePropertyComposite axisHide2;
	private ColorPicker colorPicker2;
	private Composite rangeComposite;
	private Composite domainComposite;

    /**
     * ModifyListener for the angle {@link TrackedSpinner}
     * 
     * @author Teemu Lempinen
     *
     */
    private class AngleModifyListener implements TextModifyListener, Widget {

        private ISessionContext context;
        private Object lastInput = null;

        @Override
        public void modifyText(TrackedModifyEvent e) {
            if(context == null)
                return;

            // Get the text value from spinner and associated resource (input)
            Spinner spinner = (Spinner)e.getWidget();
            final String textValue = spinner.getText();
            final Object input = lastInput;

            try {
                context.getSession().syncRequest(new WriteRequest() {

                    @Override
                    public void perform(WriteGraph graph) throws DatabaseException {
                        JFreeChartResource jfree = JFreeChartResource.getInstance(graph);
                        Resource domainAxis = AdaptionUtils.adaptToSingle(input, Resource.class);
                        try {
                            // usually reliable, since the spinner does all the checks
                            Double value = Double.parseDouble(textValue); 
                            Double oldValue = graph.getPossibleRelatedValue(domainAxis, jfree.Axis_rotateLabelDegrees, Bindings.DOUBLE);
                            if(oldValue == null || !oldValue.equals(value)) {
                                graph.claimLiteral(domainAxis, jfree.Axis_rotateLabelDegrees, value, Bindings.DOUBLE);
                                angleInt = value.intValue();
                            }
                        } catch (NumberFormatException e) {
                            graph.claimLiteral(domainAxis, jfree.Axis_rotateLabelDegrees, 0.0, Bindings.DOUBLE);
                            angleInt = 0;
                        }
                    }

                });
            } catch (DatabaseException e1) {
                e1.printStackTrace();
            }
        }

        @Override
        public void setInput(ISessionContext context, Object parameter) {
            this.context = context;
            lastInput = parameter;
        }

    }

    /**
     * Class for setting the value for angle {@link TrackedSpinner}
     * @author Teemu Lempinen
     *
     */
    private class AngleSelectionFactory extends ReadFactoryImpl<Resource, Integer>   {

        @Override
        public Integer perform(ReadGraph graph, Resource domainAxis) throws DatabaseException {
            if(angleInt == null) {
                Double angle = 0.0;
                JFreeChartResource jfree = JFreeChartResource.getInstance(graph);
                if(domainAxis != null) {
                    Double value = graph.getPossibleRelatedValue(domainAxis, jfree.Axis_rotateLabelDegrees);
                    if(value != null)
                        angle = value;
                }
                return angle.intValue();
            } else {
                return angleInt;
            }
        }

    }

    @Override
    public void setInput(final ISessionContext context, Object input) {
        final Resource chart = AdaptionUtils.adaptToSingle(input, Resource.class);
        if(chart == null)
            return; 

        context.getSession().asyncRequest(new ReadRequest() {

            @Override
            public void run(ReadGraph graph) throws DatabaseException {
                Layer0 l0 = Layer0.getInstance(graph);
                JFreeChartResource jfree = JFreeChartResource.getInstance(graph);
                Resource plot = graph.syncRequest(new PossibleObjectWithType(chart, l0.ConsistsOf, jfree.Plot));
                if(plot == null) return;
                final Resource rangeAxis = graph.getPossibleObject(plot, jfree.Plot_rangeAxis);
                final Resource domainAxis = graph.getPossibleObject(plot, jfree.Plot_domainAxis);
                if(rangeAxis == null && domainAxis == null) return;
                Display.getDefault().asyncExec(new Runnable() {
					
					@Override
					public void run() {
						if (rangeAxis != null)
							rangeAxisSupport.fireInput(context, new StructuredSelection(rangeAxis));
						if (domainAxis != null)
							domainAxisSupport.fireInput(context, new StructuredSelection(domainAxis));
						
					}
				});
            }
        });
    }

	@Override
	protected void createAndAddControls(Composite body, IWorkbenchSite site,
			ISessionContext context, WidgetSupport support) {
		support.register(this);

        // Scrolled composite containing all of the properties in this tab
        sc = new ScrolledComposite(body, SWT.NONE | SWT.H_SCROLL | SWT.V_SCROLL);
        sc.setExpandHorizontal(true);
        sc.setExpandVertical(true);

        composite = new Composite(sc, SWT.NONE);

        // Domain Axis properties
        domainGroup = new Group(composite, SWT.NONE);
        domainGroup.setText("Domain axis");

        domainComposite = new Composite(domainGroup, SWT.NONE);
        
        // Label for x-axis
        labelLabel = new Label(domainComposite, SWT.NONE);
        labelLabel.setText("Label:");

        rangelabel = new TrackedText(domainComposite, domainAxisSupport, SWT.BORDER);
        rangelabel.setTextFactory(new StringPropertyFactory(Layer0.URIs.HasLabel, ""));
        rangelabel.addModifyListener(new StringPropertyModifier(context, Layer0.URIs.HasLabel));
        rangelabel.setColorProvider(new JFreeChartPropertyColorProvider(rangelabel.getResourceManager()));
        
        angleLabel = new Label(domainComposite, SWT.NONE);
        angleLabel.setText("Label angle:");

        angleComposite = new Composite(domainComposite, SWT.NONE);
        angle = new TrackedSpinner(angleComposite, domainAxisSupport, SWT.BORDER);
        angle.setSelectionFactory(new AngleSelectionFactory());
        angle.addModifyListener(new AngleModifyListener());
        angle.setMinimum(0);
        angle.setMaximum(90);
        angle.getWidget().setIncrement(5);

        // Domain Color
        labelColor = new Label(domainComposite, SWT.NONE);
        labelColor.setText("Color:");

        colorPicker = new ColorPicker(domainComposite, context, domainAxisSupport, SWT.NONE);

        axisHide = new AxisHidePropertyComposite(domainGroup, context, domainAxisSupport, SWT.NONE);

        
        // Range Axis properties
        rangeGroup = new Group(composite, SWT.NONE);
        rangeGroup.setText("Range axis");

        rangeComposite = new Composite(rangeGroup, SWT.NONE);
        
        // Label for range axis
        labelLabel2 = new Label(rangeComposite, SWT.NONE);
        labelLabel2.setText("Label:");
        labelLabel2.setAlignment(SWT.RIGHT);

        rangelabel2 = new TrackedText(rangeComposite, rangeAxisSupport, SWT.BORDER);
        rangelabel2.setTextFactory(new StringPropertyFactory(Layer0.URIs.HasLabel, ""));
        rangelabel2.addModifyListener(new StringPropertyModifier(context, Layer0.URIs.HasLabel));
        rangelabel2.setColorProvider(new JFreeChartPropertyColorProvider(rangelabel2.getResourceManager()));


        // Min and max values for range axis
        labelMin = new Label(rangeComposite, SWT.NONE);
        labelMin.setText("Min:");

        minmax = new Composite(rangeComposite, SWT.NONE);
        rangemin = new TrackedText(minmax, rangeAxisSupport, SWT.BORDER);
        rangemin.setColorProvider(new JFreeChartPropertyColorProvider(rangemin.getResourceManager()));
        rangemin.setTextFactory(new DoublePropertyFactory(JFreeChartResource.URIs.Axis_min));
        rangemin.addModifyListener(new DoublePropertyModifier(context, JFreeChartResource.URIs.Axis_min));
        rangemin.setInputValidator(new DoubleValidator(true));

        labelMax = new Label(minmax, SWT.NONE);
        labelMax.setText("Max:");
        rangemax = new TrackedText(minmax, rangeAxisSupport, SWT.BORDER);
        rangemax.setColorProvider(new JFreeChartPropertyColorProvider(rangemax.getResourceManager()));
        rangemax.setTextFactory(new DoublePropertyFactory(JFreeChartResource.URIs.Axis_max));
        rangemax.addModifyListener(new DoublePropertyModifier(context, JFreeChartResource.URIs.Axis_max));
        rangemax.setInputValidator(new DoubleValidator(true));

        // Range Color
        labelColor2 = new Label(rangeComposite, SWT.NONE);
        labelColor2.setText("Color:");

        colorPicker2 = new ColorPicker(rangeComposite, context, rangeAxisSupport, SWT.NONE);

        axisHide2 = new AxisHidePropertyComposite(rangeGroup, context, rangeAxisSupport, SWT.NONE);
        
        // Resize scrolled composite
        sc.setContent(composite);
	}

	@Override
	protected void createControlLayoutVertical() {
		// Scrolled composite containing all of the properties in this tab
        GridDataFactory.fillDefaults().grab(true, true).applyTo(sc);
        GridLayoutFactory.fillDefaults().margins(3, 3).applyTo(sc);

        GridLayoutFactory.fillDefaults().margins(3, 3).applyTo(composite);

        // Domain Axis properties
        GridDataFactory.fillDefaults().grab(false, false).applyTo(domainGroup);
        GridLayoutFactory.fillDefaults().margins(3, 3).numColumns(1).applyTo(domainGroup);

        GridDataFactory.fillDefaults().grab(true, false).applyTo(domainComposite);
        GridLayoutFactory.fillDefaults().margins(3, 3).numColumns(2).applyTo(domainComposite);

        // Label for x-axis
        GridDataFactory.fillDefaults().align(SWT.END, SWT.CENTER).applyTo(labelLabel);

        GridDataFactory.fillDefaults().grab(true, false).span(1, 1).applyTo(rangelabel.getWidget());
        
        GridDataFactory.fillDefaults().align(SWT.END, SWT.CENTER).applyTo(angleLabel);

        GridDataFactory.fillDefaults().applyTo(angleComposite);
        GridLayoutFactory.fillDefaults().applyTo(angleComposite);
        GridDataFactory.fillDefaults().align(SWT.BEGINNING, SWT.CENTER).grab(false, true).applyTo(angle.getWidget());

        // Domain Color
        GridDataFactory.fillDefaults().grab(false, false).align(SWT.END, SWT.CENTER).applyTo(labelColor);

        GridDataFactory.fillDefaults().grab(false, true).minSize(SWT.DEFAULT, 31).align(SWT.BEGINNING, SWT.CENTER).applyTo(colorPicker);

        GridDataFactory.fillDefaults().applyTo(axisHide);

        domainGroup.layout();

        // Range Axis properties
        GridDataFactory.fillDefaults().grab(true, false).applyTo(rangeGroup);
        GridLayoutFactory.fillDefaults().margins(3, 3).numColumns(1).applyTo(rangeGroup);

        GridDataFactory.fillDefaults().grab(true, false).applyTo(rangeComposite);
        GridLayoutFactory.fillDefaults().margins(3, 3).numColumns(2).applyTo(rangeComposite);

        // Label for range axis
        GridDataFactory.fillDefaults().hint(angleLabel.getBounds().width, SWT.DEFAULT).align(SWT.END, SWT.CENTER).applyTo(labelLabel2);

        GridDataFactory.fillDefaults().grab(true, false).span(1, 1).applyTo(rangelabel2.getWidget());

        // Min and max values for range axis
        GridDataFactory.fillDefaults().align(SWT.END, SWT.CENTER).applyTo(labelMin);

        GridDataFactory.fillDefaults().align(SWT.BEGINNING, SWT.CENTER).grab(false, true).applyTo(minmax);
        GridLayoutFactory.fillDefaults().numColumns(3).applyTo(minmax);
        GridDataFactory.fillDefaults().hint(50, SWT.DEFAULT).applyTo(rangemax.getWidget());
        GridDataFactory.fillDefaults().hint(50, SWT.DEFAULT).applyTo(rangemin.getWidget());
        
        // Range Color
        GridDataFactory.fillDefaults().grab(false, false).align(SWT.END, SWT.CENTER).applyTo(labelColor2);

        GridDataFactory.fillDefaults().grab(false, true).minSize(SWT.DEFAULT, 31).align(SWT.BEGINNING, SWT.CENTER).applyTo(colorPicker2);

        GridDataFactory.fillDefaults().applyTo(axisHide2);

        // Resize scrolled composite
        Point size = composite.computeSize(SWT.DEFAULT, SWT.DEFAULT);
        sc.setMinSize(size);
	}

	@Override
	protected void createControlLayoutHorizontal(boolean wideScreen) {
		// Scrolled composite containing all of the properties in this tab
        GridDataFactory.fillDefaults().grab(true, true).applyTo(sc);
        GridLayoutFactory.fillDefaults().margins(3, 3).applyTo(sc);

        GridLayoutFactory.fillDefaults().margins(3, 3).applyTo(composite);

        // Domain Axis properties
        GridDataFactory.fillDefaults().grab(true, false).applyTo(domainGroup);
        GridLayoutFactory.fillDefaults().margins(3, 3).numColumns(2).applyTo(domainGroup);

        GridDataFactory.fillDefaults().grab(true, false).applyTo(domainComposite);
        GridLayoutFactory.fillDefaults().numColumns(4).applyTo(domainComposite);

        // Label for x-axis
        GridDataFactory.fillDefaults().align(SWT.END, SWT.CENTER).applyTo(labelLabel);

        GridDataFactory.fillDefaults().grab(true, false).span(3, 1).applyTo(rangelabel.getWidget());
        
        GridDataFactory.fillDefaults().align(SWT.END, SWT.CENTER).applyTo(angleLabel);

        GridDataFactory.fillDefaults().applyTo(angleComposite);
        GridLayoutFactory.fillDefaults().applyTo(angleComposite);
        GridDataFactory.fillDefaults().align(SWT.BEGINNING, SWT.CENTER).grab(false, true).applyTo(angle.getWidget());

        // Domain Color
        GridDataFactory.fillDefaults().grab(true, false).align(SWT.END, SWT.CENTER).applyTo(labelColor);

        GridDataFactory.fillDefaults().grab(false, true).minSize(SWT.DEFAULT, SWT.DEFAULT).align(SWT.BEGINNING, SWT.CENTER).applyTo(colorPicker);

        GridDataFactory.fillDefaults().applyTo(axisHide);

        domainGroup.layout();

        // Range Axis properties
        GridDataFactory.fillDefaults().grab(true, false).applyTo(rangeGroup);
        GridLayoutFactory.fillDefaults().margins(3, 3).numColumns(2).applyTo(rangeGroup);

        GridDataFactory.fillDefaults().grab(true, false).applyTo(rangeComposite);
        GridLayoutFactory.fillDefaults().numColumns(4).applyTo(rangeComposite);

        // Label for range axis
        GridDataFactory.fillDefaults().hint(angleLabel.getBounds().width, SWT.DEFAULT).align(SWT.END, SWT.CENTER).applyTo(labelLabel2);

        GridDataFactory.fillDefaults().grab(true, false).span(3, 1).applyTo(rangelabel2.getWidget());

        // Min and max values for range axis
        GridDataFactory.fillDefaults().align(SWT.END, SWT.CENTER).applyTo(labelMin);

        GridDataFactory.fillDefaults().align(SWT.BEGINNING, SWT.CENTER).grab(false, true).applyTo(minmax);
        GridLayoutFactory.fillDefaults().numColumns(3).applyTo(minmax);
        GridDataFactory.fillDefaults().hint(SWT.DEFAULT, SWT.DEFAULT).applyTo(rangemax.getWidget());
        GridDataFactory.fillDefaults().hint(SWT.DEFAULT, SWT.DEFAULT).applyTo(rangemin.getWidget());
        
        // Range Color
        GridDataFactory.fillDefaults().grab(true, false).align(SWT.END, SWT.CENTER).applyTo(labelColor2);

        GridDataFactory.fillDefaults().grab(false, true).minSize(SWT.DEFAULT, SWT.DEFAULT).align(SWT.BEGINNING, SWT.CENTER).applyTo(colorPicker2);

        GridDataFactory.fillDefaults().applyTo(axisHide2);

        // Resize scrolled composite
        Point size = composite.computeSize(SWT.DEFAULT, SWT.DEFAULT);
        sc.setMinSize(size);
	}
}
