/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.jfreechart.chart.properties;

import org.eclipse.core.runtime.ListenerList;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Spinner;
import org.simantics.browsing.ui.common.ErrorLogger;
import org.simantics.browsing.ui.swt.widgets.impl.ReadFactory;
import org.simantics.browsing.ui.swt.widgets.impl.TextModifyListener;
import org.simantics.browsing.ui.swt.widgets.impl.TrackedModifyEvent;
import org.simantics.browsing.ui.swt.widgets.impl.Widget;
import org.simantics.browsing.ui.swt.widgets.impl.WidgetSupport;
import org.simantics.db.management.ISessionContext;
import org.simantics.db.procedure.Listener;
import org.simantics.utils.ui.SWTUtils;

/**
 * Class for implementing Widget behavior for SWT Spinner in Simantics.
 * 
 * @author Teemu Lempinen
 *
 */
public class TrackedSpinner implements Widget {

    private Spinner         spinner;
    private ListenerList    modifyListeners;
    private ReadFactory<?, Integer> selectionFactory;


    public TrackedSpinner(Composite parent, WidgetSupport support, int style) {
        spinner = new Spinner(parent, style);
        support.register(this);
        
        // Add a ModifyListener that uses all listeners in modifyListeners -list
        spinner.addModifyListener(new ModifyListener() {
            
            @Override
            public void modifyText(ModifyEvent e) {
                if (modifyListeners != null) {
                    TrackedModifyEvent event = new TrackedModifyEvent(spinner, spinner.getText());
                    for (Object o : modifyListeners.getListeners()) {
                        ((TextModifyListener) o).modifyText(event);
                    }
                }                
            }
        });
    }

    @Override
    public void setInput(ISessionContext context, Object input) {

        // Update all modifyListeners
        if (modifyListeners != null) {
            for (Object o : modifyListeners.getListeners()) {
                if(o instanceof Widget) {
                    ((Widget) o).setInput(context, input);
                }
            }
        }

        if (selectionFactory != null) {
            // Get a value for the spinner
            selectionFactory.listen(context, input, new Listener<Integer>() {
                @Override
                public void exception(Throwable t) {
                    ErrorLogger.defaultLogError(t);
                }
                @Override
                public void execute(final Integer selection) {
                    SWTUtils.asyncExec(spinner, new Runnable() {
                        @Override
                        public void run() {
                            if(isDisposed()) return;
                            spinner.setSelection(selection);
                        }
                    });
                }
                @Override
                public boolean isDisposed() {
                    return spinner.isDisposed();
                }
            });
        }

    }
    
    /**
     * Set a selection factory for the spinner
     * 
     * @param selectionFactory ReadFactory<?, Integer> SelectionFactory
     */
    public void setSelectionFactory(ReadFactory<?, Integer> selectionFactory) {
        this.selectionFactory = selectionFactory;
    }
    
    /**
     * Add a modifyListener for the spinner
     * @param listener TextModifyListener
     */
    public synchronized void addModifyListener(TextModifyListener listener) {
        if (modifyListeners == null) {
            modifyListeners = new ListenerList(ListenerList.IDENTITY);
        }
        modifyListeners.add(listener);
    }

    /**
     * Remove modifyListener from the spinner
     * 
     * @param listener TextModifyListener
     */
    public synchronized void removeModifyListener(TextModifyListener listener) {
        if (modifyListeners == null)
            return;
        modifyListeners.remove(listener);
    }

    /**
     * Get the SWT Spinner of this TrackedSpinner widget
     * @return
     */
    public Spinner getWidget() {
        return spinner;
    }
    
    /**
     * Set minimum value 
     * @param value int minimum value
     */
    public void setMinimum(int value) {
        spinner.setMinimum(value);
    }
    
    /**
     * Set maximum value
     * @param value int maximum value
     */
    public void setMaximum(int value) {
        spinner.setMaximum(value);
    }
    
    /**
     * Sets the receiver's selection, minimum value, maximum value, digits, increment and page increment all at once. 
     * 
     * @param selection the new selection value
     * @param minimum the new minimum value
     * @param maximum the new maximum value
     * @param digits the new digits value
     * @param increment the new increment value
     * @param pageIncrement the new pageIncrement value
     */
    public void setValues(int selection, int  minimum, int  maximum, int digits, int increment, int pageIncrement) {
        spinner.setValues(selection, minimum, maximum, digits, increment, pageIncrement);
    }
    
    /**
     * Sets the selection, which is the receiver's position, to the argument.
     * If the argument is not within the range specified by minimum and maximum, 
     * it will be adjusted to fall within this range.
     * 
     * @param value
     */
    public void setSelection(int value) {
        spinner.setSelection(value);
    }

}
