package org.simantics.jfreechart.chart;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Paint;
import java.util.ArrayList;

import javax.swing.SwingUtilities;

import org.jfree.chart.ChartColor;
import org.jfree.chart.labels.StandardXYToolTipGenerator;
import org.jfree.chart.plot.DefaultDrawingSupplier;
import org.jfree.chart.renderer.AbstractRenderer;
import org.jfree.chart.renderer.xy.AbstractXYItemRenderer;
import org.jfree.chart.renderer.xy.XYLineAndShapeRenderer;
import org.jfree.data.general.Dataset;
import org.jfree.data.xy.DefaultXYDataset;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.procedure.adapter.DisposableListener;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.request.Read;
import org.simantics.sysdyn.JFreeChartResource;
import org.simantics.utils.datastructures.Pair;

public class ConstantXYDataSet extends AbstractDataset implements XYDataset {

	protected IRenderer renderer;

	public ConstantXYDataSet(final Resource datasetResource) throws DatabaseException {
		super(datasetResource);
	}

	protected Dataset dataset;
	protected DisposableListener<?> datasetListener;

	@SuppressWarnings("unchecked")
	@Override
	public Dataset getDataset() {
		
		if(dataset == null) {
			dataset = new DefaultXYDataset();
		}   

		if(datasetListener == null || datasetListener.isDisposed()) {
			datasetListener = new DataSetListener();
			Simantics.getSession().asyncRequest(
					new ConstantXYDatasetRequest(resource), 
					(DisposableListener<Pair<ArrayList<ConstantXYDatasetSeries>, IRenderer>>) datasetListener); 

		}

		return dataset;
		
	}

	private class DataSetListener extends DisposableListener<Pair<ArrayList<ConstantXYDatasetSeries>, IRenderer>> {

		@Override
		public void execute(Pair<ArrayList<ConstantXYDatasetSeries>, IRenderer> result) {
			final ArrayList<ConstantXYDatasetSeries> series = result.first;
			renderer = result.second;

			// Modify series in AWT thread to avoid synchronization problems
			SwingUtilities.invokeLater(new Runnable() {

				@Override
				public void run() {

					if(dataset == null || !(dataset instanceof DefaultXYDataset))
						return;

					DefaultXYDataset ds = (DefaultXYDataset)dataset;
					org.jfree.chart.plot.XYPlot plot = ((AbstractXYItemRenderer)getRenderer()).getPlot();

					if(plot != null) {
						/*
						 *  Drawing supplier with a modified first yellow. The default first
						 *  yellow is too light to be visible against a white background
						 */
						Paint[] paintSequence = ChartColor.createDefaultPaintArray();
						paintSequence[3] = new Color(0xFF, 0xDD, 0x00);
						DefaultDrawingSupplier drawingsupplier = new DefaultDrawingSupplier(
								paintSequence,
								DefaultDrawingSupplier.DEFAULT_FILL_PAINT_SEQUENCE,
								DefaultDrawingSupplier.DEFAULT_OUTLINE_PAINT_SEQUENCE,
								DefaultDrawingSupplier.DEFAULT_STROKE_SEQUENCE,
								DefaultDrawingSupplier.DEFAULT_OUTLINE_STROKE_SEQUENCE,
								DefaultDrawingSupplier.DEFAULT_SHAPE_SEQUENCE);
						plot.setDrawingSupplier(drawingsupplier);
					}
					// Remove all series
					for(int i = ds.getSeriesCount() - 1; i >= 0; i-- ) {
						ds.removeSeries(ds.getSeriesKey(i));
					}

					// Add found series
					for(int i = 0; i < series.size(); i++) {
						ConstantXYDatasetSeries s = series.get(i);
						String name = s.name;
						if(ds.indexOf(name) >= 0)
							name = name + (i + 1);
						ds.addSeries(name, s.values);
						getRenderer().setSeriesStroke(i, new BasicStroke((float)s.width));
						getRenderer().setSeriesPaint(i, s.color);
					} 
				}
			});
		}

		@Override
		public void exception(Throwable t) {
			t.printStackTrace();
		}

	}


	@Override
	public void dispose() {
		if(datasetListener != null) {
			datasetListener.dispose();
			datasetListener = null;
		}
	}


	@Override
	public AbstractRenderer getRenderer() {
		if(renderer == null) {
			try {
				renderer = Simantics.getSession().syncRequest(new Read<IRenderer>() {

					@Override
					public IRenderer perform(ReadGraph graph) throws DatabaseException {
						JFreeChartResource jfree = JFreeChartResource.getInstance(graph);
						IRenderer renderer = null;
						Resource rendererResource = graph.getPossibleObject(resource, jfree.Dataset_renderer);
						if(rendererResource != null)
							renderer = graph.adapt(rendererResource, IRenderer.class);
						return renderer;
					}
				});
			} catch (DatabaseException e) {
			}
			if(renderer == null) {
				XYLineAndShapeRenderer renderer = new XYLineAndShapeRenderer(true, false);
				renderer.setDefaultToolTipGenerator(new StandardXYToolTipGenerator());
				return renderer;
			} else {
				return renderer.getRenderer();
			}
		} else {
			return renderer.getRenderer();
		}
	}

}
