/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.jfreechart;

import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.swt.SWT;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.DropTarget;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.simantics.db.Resource;
import org.simantics.jfreechart.chart.ChartComposite;
import org.simantics.ui.dnd.LocalObjectTransfer;

/**
 * This class represents an expanded chart element in {@link ChartPanel}. It contains 
 * a header {@link ChartPanelHeader} and the actual chart.
 * 
 * @author Teemu Lempinen
 *
 */
public class ChartPanelElement extends Composite {

    public static int CHART_MINIMUM_WIDTH = 300;
    public static int CHART_MINIMUM_HEIGHT = 150;

    private ChartPanel panel;
    private ChartComposite chartComposite;
    private Resource chartResource;
    private boolean minimized = false;

    public Resource getResource() {
        return chartResource;
    }

    @Override
    public Object getLayoutData () {
        checkWidget();
        Object oldData = super.getLayoutData();
        if(oldData == null || !(oldData instanceof GridData)) {
            oldData = GridDataFactory.fillDefaults().create();
        }

        int size = panel.getElements().size();
        GridData data = (GridData) oldData;
        // Horizontal data
        data.widthHint = CHART_MINIMUM_WIDTH;
        if(getResource() == null && size == 1) {
            data.grabExcessHorizontalSpace = true;
        } else if(getResource() == null && !panel.isVertical()){
            data.grabExcessHorizontalSpace = false;
            data.widthHint = SWT.DEFAULT;
        } else if(minimized && !panel.isVertical()) {
            data.grabExcessHorizontalSpace = false;
            data.widthHint = SWT.DEFAULT;
        } else {
            data.grabExcessHorizontalSpace = true;
        }

        // Vertical data
        if(getResource() == null && size == 1) {
            data.grabExcessVerticalSpace = true;
        } else if(!minimized && getResource() != null) {
            data.grabExcessVerticalSpace = true;
            data.heightHint = CHART_MINIMUM_HEIGHT;
        } else if(!panel.isVertical()){
            data.grabExcessVerticalSpace = true;
        } else {
            data.grabExcessVerticalSpace = false;
            data.heightHint = SWT.DEFAULT;
        }
        return data;
    }

    /**
     * Creates an expanded chart panel element into parent composite.
     * 
     * @param parent The parent composite where the chart element is created
     * @param panel The {@link ChartPanel} containing the chart element
     * @param name The name of the chart
     * @param style The Style of the created chart element
     */
    public ChartPanelElement(Composite parent, ChartPanel panel, Resource chartResource, int style) {
        this(parent, panel, chartResource, false, style);
    }

    /**
     *  Creates a chart panel element into parent composite.
     * @param parent The parent composite where the chart element is created
     * @param panel The {@link ChartPanel} containing the chart element
     * @param name The name of the chart
     * @param minimized Is the chart-section minimized
     * @param style The Style of the created chart element
     */
    public ChartPanelElement(Composite parent, ChartPanel panel, Resource chartResource, boolean minimized, int style) {
        super(parent, style | SWT.NONE );
        
        this.panel = panel;
        this.chartResource = chartResource;
        this.minimized = minimized;

        if(panel.isVertical() || getResource() == null)
            GridLayoutFactory.fillDefaults().spacing(0,0).applyTo(this);
        else
            GridLayoutFactory.fillDefaults().numColumns(2).spacing(0,0).applyTo(this);

        GridDataFactory.fillDefaults().applyTo(this);

        // Separator for dropping other elements
        ChartPanelSeparator separator = new ChartPanelSeparator(this, panel, this, SWT.NONE);

        if (chartResource != null) {
            Composite c = new Composite(this, SWT.NONE);
            GridDataFactory.fillDefaults().grab(true, true).applyTo(c);
            GridLayoutFactory.fillDefaults().spacing(0, 0).applyTo(c);
            // Header
            new ChartPanelHeader(c, this, chartResource, SWT.BORDER);

            // Chart
            chartComposite = new ChartComposite(c, chartResource, SWT.BORDER);
        }
        
        DropTarget target = new DropTarget(this, DND.DROP_COPY | DND.DROP_MOVE);
        target.setTransfer(new Transfer[] {  LocalObjectTransfer.getTransfer() });
        target.addDropListener(new ChartDropTarget(separator, this, panel));

    }

    /**
     * Returns the chart resource associated with this element
     * @return chart resource
     */
    public Resource getChartResource() {
        return this.chartResource;
    }

    /**
     * Returns the minimized state of this element
     * @return is the element minimized
     */
    public boolean isMinimized() {
        return minimized;
    }

    /**
     * Change the minimized state of this element
     */
    public void toggleMinimize() {
        toggleMinimize(false);
    }
    /**
     * Change the minimized state of this element
     */
    public void toggleMinimize(boolean callSave) {
        minimized = Boolean.FALSE.equals(minimized);
        GridData data = (GridData) chartComposite.getLayoutData();
        if(panel.isVertical())
            data.exclude = minimized;
        else
            data.exclude = false;
        chartComposite.setVisible(!minimized);

        Composite parent = getParent();
        data = (GridData) getLayoutData();
        GridData parentData = (GridData)parent.getLayoutData();
        parentData.grabExcessHorizontalSpace = data.grabExcessHorizontalSpace;
        parentData.grabExcessVerticalSpace = data.grabExcessVerticalSpace;
        parentData.heightHint = data.heightHint;

        if(callSave) {
            panel.saveState();
        }
        panel.layout();
    }

    /**
     * Remove this chart panel element from its panel
     */
    public void remove() {
        panel.removeChart(chartResource);
    }
}
