package org.simantics.graph.refactoring;

import java.io.BufferedInputStream;
import java.io.DataInputStream;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;

import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.container.DataContainer;
import org.simantics.databoard.container.DataContainers;
import org.simantics.graph.representation.PrettyPrintTG;
import org.simantics.graph.representation.TransferableGraph1;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Antti Villberg
 * @since 1.24.0
 */
public class FixExportedOntology {

	private static final Logger LOGGER = LoggerFactory.getLogger(FixExportedOntology.class);

	static TransferableGraph1 convertExportedSharedOntologyIntoBundleOntology(Path input, Path output) throws Exception {
		LOGGER.info("Converting exported shared ontology\n\t{}\nto bundle-compatible ontology\n\t{}", input.toString(), output.toString());
		try (InputStream is = new BufferedInputStream(Files.newInputStream(input), 128*1024)) {
			Binding tgBinding = TransferableGraph1.BINDING;
			DataContainer container = DataContainers.readFile(new DataInputStream(is), tgBinding); 
			TransferableGraph1 graph = (TransferableGraph1) container.content.getValue(tgBinding);

			GraphRefactoringUtils.fixOntologyExport(graph);
			container = TransferableGraphHasher.addHashToTG(container, graph);

			DataContainers.writeFile(output.toFile(), container);
			return graph;
		}
	}

	private static Path replaceExtension(Path p, String newExtension) {
		String newName = p.getFileName().toString();
		int lastDot = newName.lastIndexOf('.');
		if (lastDot > -1)
			newName = newName.substring(0, lastDot);
		return p.resolveSibling(newName + newExtension);
	}

	private static void createTg(Path input, Path output) throws Exception {
		convertExportedSharedOntologyIntoBundleOntology(input, output);
	}

	public static void createTg(Path input) throws Exception {
		createTg(input, replaceExtension(input, ".tg"));
	}

	public static void createTGAndPGraph(Path input) throws Exception {
		createTGAndPGraph(input, true);
	}

	public static void createTGAndPGraph(Path input, boolean writePGraph) throws Exception {
		TransferableGraph1 tg = convertExportedSharedOntologyIntoBundleOntology(input, replaceExtension(input, ".tg"));
		if (writePGraph) {
			String listing = PrettyPrintTG.print(tg, false);
			Files.write(replaceExtension(input, ".pgraph"), listing.getBytes(),StandardOpenOption.CREATE);
		}
	}

	public static void main(String[] args) throws Exception {
		if (args.length == 0) {
			System.out.println("Required arguments: <input .sharedLibrary file> [<output .tg file>]");
		} else if (args.length == 1) {
			Path input = Paths.get(args[0]);
			createTGAndPGraph(input);
		} else {
			convertExportedSharedOntologyIntoBundleOntology(Paths.get(args[0]), Paths.get(args[1]));
		}
	}

}
