package org.simantics.graph.query;

import gnu.trove.list.array.TIntArrayList;
import gnu.trove.map.hash.THashMap;
import gnu.trove.set.hash.THashSet;
import gnu.trove.set.hash.TIntHashSet;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;

import org.simantics.databoard.adapter.AdaptException;
import org.simantics.databoard.adapter.RuntimeAdaptException;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.mutable.Variant;
import org.simantics.databoard.type.Datatype;
import org.simantics.graph.store.GraphStore;
import org.simantics.graph.store.IdRes;
import org.simantics.graph.store.IdentityStore;
import org.simantics.graph.store.PathPattern;
import org.simantics.graph.store.StatementStore;

public class CompositeGraph implements IGraph {

	ArrayList<GraphStore> fragments = new ArrayList<GraphStore>();
	Paths paths;
	
	public CompositeGraph(Paths paths) {
        this.paths = paths;
    }

    public void addFragment(GraphStore fragment) {
		fragments.add(fragment);
	}
	
	public void undoAddFragment() {
		fragments.remove(fragments.size()-1);
	}
	
	public void addFragments(Collection<GraphStore> fragments) {
		fragments.addAll(fragments);
	}
	
	protected void rawGetObjects(GraphStore fragment, int subject, Res predicate, Collection<Res> result) {
		int predicateId;
		if(predicate instanceof Path) {
			predicateId = fragment.identities.pathToId((Path)predicate);
			if(predicateId < 0)
				return;
		} 
		else {
			IdRes idPredicate = (IdRes)predicate;
			if(idPredicate.fragment != fragment)
				return;
			predicateId = idPredicate.id;
		}
		
		TIntArrayList objects = fragment.statements.getObjects(subject, predicateId);
		fragment.addIdsToResult(objects, result);
	}
			
	private void rawGetObjects(Res subject, Res predicate, Collection<Res> result) {
		if(subject instanceof Path) {
			Path path = (Path)subject;
			for(GraphStore fragment : fragments) {
				int id = fragment.identities.pathToId(path);
				if(id >= 0)
					rawGetObjects(fragment, id, predicate, result);
			}
		}
		else {
			IdRes idRes = (IdRes)subject;						
			rawGetObjects(idRes.fragment, idRes.id, predicate, result);
		}
	}
	
	public boolean hasRawObjects(Res subject, Path predicate) {
		ArrayList<Res> result = new ArrayList<Res>();
		rawGetObjects(subject, predicate, result);
		return !result.isEmpty();
	}
	
	@Override
	public Collection<Res> rawGetObjects(Res subject, Res predicate) {
		ArrayList<Res> result = new ArrayList<Res>();
		rawGetObjects(subject, predicate, result);
		return result;
	}
	
	@Override
	public Res singleRawObject(Res subject, Res predicate) throws NoUniqueObjectException {
		ArrayList<Res> result = new ArrayList<Res>(1);
		rawGetObjects(subject, predicate, result);
		if(result.size() != 1)
			throw new NoUniqueObjectException("No unique objects (" + result.size()+ ") for " +
					subject + " -> " + predicate);		
		return result.get(0);
	}
	
	@Override
	public Collection<Res> getTypes(Res subject) {
		THashSet<Res> result = new THashSet<Res>(); 
		rawGetObjects(subject, paths.InstanceOf, result);
		for(Res type : result.toArray(new Res[result.size()]))
			collectSupertypes(type, result);
		return result;
	}
	
	public Collection<Res> getSupertypes(Res subject) {
		THashSet<Res> result = new THashSet<Res>(); 
		result.add(subject);		
		collectSupertypes(subject, result);
		return result;
	}
	
	private void collectSupertypes(Res type, THashSet<Res> result) {
		for(Res supertype : rawGetObjects(type, paths.Inherits))
			if(result.add(supertype))
				collectSupertypes(supertype, result);		
	}

	@Override
	public Collection<Res> getObjects(Res subject, Res predicate) {
		ArrayList<Res> result = new ArrayList<Res>();
		rawGetObjects(subject, predicate, result);
		for(Res type : getTypes(subject)) 
			for(Res assertion : rawGetObjects(type, paths.Asserts)) {
				Res pred = singleRawObject(assertion, paths.HasPredicate);
				if(equals(pred, predicate))
					result.add(singleRawObject(assertion, paths.HasObject));
			}
		return result;
	}
	
	public Collection<Res> getAssertedObjects(Res subject, Path predicate) {
		ArrayList<Res> result = new ArrayList<Res>();
		for(Res type : getSupertypes(subject)) 
			for(Res assertion : rawGetObjects(type, paths.Asserts)) {
				Res pred = singleRawObject(assertion, paths.HasPredicate);
				if(equals(pred, predicate))
					result.add(singleRawObject(assertion, paths.HasObject));
			}
		return result;
	}
	
	private static boolean equals(Res r1, Res r2) {
		return r1.equals(r2);
	}
	
	interface ResourceProcedure {
		public void execute(GraphStore fragment, int id);
	}
	
	interface ResourceFunction<T> {
		public T execute(GraphStore fragment, int id);
	}
	
	public void forEachFragmentContaining(Res resource, ResourceProcedure proc) {
		if(resource instanceof Path) {
			Path path = (Path)resource;
			for(GraphStore fragment : fragments) {
				int id = fragment.identities.pathToId(path);
				if(id >= 0)
					proc.execute(fragment, id);
			}
		}
		else {
			IdRes res = (IdRes)resource;
			proc.execute(res.fragment, res.id);
		}
	}
	
	public <T> T apply(Res resource, ResourceFunction <T> func) {
		if(resource instanceof Path) {
			Path path = (Path)resource;
			for(GraphStore fragment : fragments) {
				int id = fragment.identities.pathToId(path);
				if(id >= 0) {
					T value = func.execute(fragment, id);
					if(value != null)
						return value;
				}
			}
			return null;
		}
		else {
			IdRes res = (IdRes)resource;
			return func.execute(res.fragment, res.id);
		}
	}
	
	private static ResourceFunction<Datatype> getDatatype = 
		new ResourceFunction<Datatype>() {
			@Override
			public Datatype execute(GraphStore fragment, int id) {
				return fragment.values.getDatatypeValue(id);
			}		
	};
	
	private static ResourceFunction<Variant> getValue = 
		new ResourceFunction<Variant>() {
			@Override
			public Variant execute(GraphStore fragment, int id) {
				return fragment.values.getByteValue(id);
			}		
	};
		
	THashMap<Res, Datatype> datatypeCache = new THashMap<Res, Datatype>();
	
	@Override
	public Datatype getDatatype(Res resource) {
		for(Res dt : getObjects(resource, paths.HasDatatype)) {
			Datatype type = datatypeCache.get(dt);
			if(type == null) {
				type = apply(dt, getDatatype);
				datatypeCache.put(dt, type);
			}
			return type;
		}
		return null;
	}
	
	@Override
	public Datatype getAssertedDatatype(Res resource) {
		for(Res dt : getAssertedObjects(resource, paths.HasDatatype)) {
			Datatype type = datatypeCache.get(dt);
			if(type == null) {
				type = apply(dt, getDatatype);
				datatypeCache.put(dt, type);
			}
			return type;
		}
		return null;
	}
	
	@Override
	public Variant getValue(Res resource) {
		return apply(resource, getValue);
	}
	
	@Override
	public Object getValue(Res resource, Binding binding) throws NoValueException {
	    Variant value = getValue(resource);
		if(value == null)
			throw new NoValueException();
		try {
            return value.getValue(binding);
        } catch (AdaptException e) {
            throw new RuntimeAdaptException(e);
        }
	}
	
	@Override
	public void setValue(Res resource, Object value, Binding binding) {
	    final Variant variant = new Variant(binding, value);
	    apply(resource, new ResourceFunction<Object>() {
            @Override
            public Object execute(GraphStore fragment, int id) {
                fragment.values.setValue(id, variant);
                return null;
            }	        
	    });
	}
	
	/**
	 * Tells in how many fragments the resource occurs.
	 */
	public int countOccurences(Res resource) {
		return countOccurences(resource, false);
	}
		
	public int countOccurences(Res resource, boolean ignoreExternals) {
		
		if(resource instanceof IdRes)
			return 1;
		else if(resource instanceof Path) {
			Path path = (Path)resource;
			int count = 0;
			for(GraphStore fragment : fragments)
				if(fragment.identities.contains(path)) {
					if(ignoreExternals) {
						if(!fragment.identities.isNewResource(fragment.identities.pathToId(path))) {
							// This reference is external and can be ignored
							continue;
						}
					}
					++count;
				}
			return count;
		}
		else
			return 0;
	}
	
	private void collectSubtypes(THashSet<Res> types, Res type) {
		if(types.add(type)) 
			for(Res subtype : rawGetObjects(type, paths.SupertypeOf))
				collectSubtypes(types, subtype);
	}
	
	@Override
	public Collection<Res> getInstances(Res supertype) {
		THashSet<Res> types = new THashSet<Res>(); 
		collectSubtypes(types, supertype);
		
		ArrayList<Res> result = new ArrayList<Res>();
		fragmentLoop:
		for(GraphStore fragment : fragments) {
			IdentityStore identities = fragment.identities;
			StatementStore statements = fragment.statements;
			
			TIntHashSet ids = new TIntHashSet(types.size());
			for(Res type : types) {
				if(type instanceof Path) {
					int id = identities.pathToId((Path)type);
					if(id >= 0)
						ids.add(id);
				}
				else {
					IdRes idRes = (IdRes)type;
					if(idRes.fragment == fragment)
						ids.add(idRes.id);
				}
			}
			if(ids.isEmpty())
				continue;
			
			int instanceOfId = identities.pathToId(paths.InstanceOf);
			if(instanceOfId < 0)
				continue;
			
			int resourceCount = identities.getResourceCount();
			nextResource:
			for(int i=0;i<resourceCount;++i)
				for(int typeId : statements.getObjects(i, instanceOfId).toArray())
					if(ids.contains(typeId)) {
						result.add(fragment.idToRes(i));
						continue nextResource;
					}
		}
		
		return result;
	}

	@Override
	public Collection<Res> getChildren(Res res) {
		if(res instanceof Path) {
			THashSet<Res> result = new THashSet<Res>();
			for(GraphStore store : fragments) {
				IdentityStore ids = store.identities;
				int id = ids.pathToId((Path)res);
				if(id >= 0) {
					for(int child : ids.getChildren(id))
						result.add(store.idToRes(child));
				}
			}
			return result;
		}
		else
			return Collections.<Res>emptyList();
	}
	
	public Collection<Path> searchByPattern(String pattern) {
		THashSet<Path> result = new THashSet<Path>();
		PathPattern pathPattern = PathPattern.compile(pattern);
		for(GraphStore store : fragments)
			pathPattern.search(store.identities, result);
		return result;
	}
	
	@Override
	public Paths getPaths() {
	    return paths;
	}
}
