package org.simantics.graph.db;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.channels.ReadableByteChannel;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.simantics.db.ReadGraph;
import org.simantics.graph.query.Path;
import org.simantics.graph.query.PathChild;
import org.simantics.graph.query.TransferableGraphConversion;
import org.simantics.graph.query.UriUtils;
import org.simantics.graph.refactoring.GraphRefactoringUtils;
import org.simantics.graph.representation.External;
import org.simantics.graph.representation.Identity;
import org.simantics.graph.representation.Root;
import org.simantics.graph.representation.TransferableGraph1;
import org.simantics.graph.representation.Value;
import org.simantics.graph.store.IdentityStore;
import org.simantics.utils.datastructures.BijectionMap;

/**
 * Namespace converting StreamingTransferableGraphFileReader
 * 
 * Notes:
 *   Name space conversion has not been tested, and it may not work at all.
 *   Reported getStatementCount() does not take account possible filtering.
 *   All Identities are cached, because underlaying streams cannot be reset. (StreamingTransferableGraphImportProcess does the same thing).
 *    
 * 
 * @author MarkoLuukkainen
 *
 */
public class AdaptingTransferableGraphFileReader extends StreamingTransferableGraphFileReader{

	Map<String,String> identityMap = new HashMap<>();
	Set<String> ignoreSet = new HashSet<>();
	
	public AdaptingTransferableGraphFileReader(File file, boolean deleteOnClose) throws Exception {
		super(file, deleteOnClose);
	}

	public AdaptingTransferableGraphFileReader(File file, int size) throws IOException {
		super(file, size);
	}

	public AdaptingTransferableGraphFileReader(File file) throws Exception {
		super(file);
	}

	public AdaptingTransferableGraphFileReader(InputStream stream, int size) throws IOException {
		super(stream, size);
	}

	public AdaptingTransferableGraphFileReader(InputStream stream) throws Exception {
		super(stream);
	}

	public AdaptingTransferableGraphFileReader(ReadableByteChannel channel, int size) throws IOException {
		super(channel, size);
	}

	public AdaptingTransferableGraphFileReader(ReadableByteChannel channel) throws Exception {
		super(channel);
	}
	
	@Override
	public TransferableGraphSource readTG() throws Exception {
		if(getSize() == 0) return null;

		return new AdaptingFileTransferableGraphSource();
	}
	
	public void setIdentityMap(Map<String, String> identityMap) {
		this.identityMap = identityMap;
	}
	
	public void setIgnoreSet(Set<String> ignoreSet) {
		this.ignoreSet = ignoreSet;
	}
	
	public static void unfixIncorrectRoot(List<Identity> ids) {
        for(int i=0;i<ids.size();++i) {
            Identity id = ids.get(i);
            if(id.definition instanceof Root) {
                Root root = (Root)id.definition;
                if(root.name.equals("") && root.type.equals("")) {
                    id.definition = new External(-1, "http:/");
                    return;
                }
            }
        }
    }
	
	class AdaptingFileTransferableGraphSource extends FileTransferableGraphSource {

		boolean init;
		
		List<Identity> newIdentities;
		int maxId = 0;
		int oldIdentitiesCount;
		
		Set<Integer> toIgnore = new HashSet<>();
		Map<Integer,Integer> toSwap = new HashMap<>();
		
		public AdaptingFileTransferableGraphSource() throws Exception {
			super();
			init = !(identityMap.size() > 0 || ignoreSet.size() > 0);			
		}
		
		public void init(ReadGraph graph) throws Exception {
			if (init)
				return;
			
			List<Identity> oldIdentities = new ArrayList<>();
			super.getIdentityCount();
			// Note: We were attempting to preserve streaming capabilities of the original implementation, and caching only external and root identities.
			//       The problem is that underlaying stream does not support mark or reset, and reads must be done in defined order.       
			//       Additionally, StreamingTransferableGraphImportProcess caches all identities, so streaming identities is unnecessary.
			//super.dis.mark(-1);
//			super.forIdentities(graph, new TransferableGraphSourceProcedure<Identity>() {
//				@Override
//				public void execute(Identity value) throws Exception {
//					if (value.definition instanceof External) {
//						oldIdentities.add(value);
//					} else if (value.definition instanceof Root) {
//						oldIdentities.add(value);
//					} 
//					maxId = Math.max(maxId, value.resource);
//				}
//			});
//			super.dis.reset();
			super.forIdentities(graph, new TransferableGraphSourceProcedure<Identity>() {
				@Override
				public void execute(Identity value) throws Exception {
					oldIdentities.add(value);
					maxId = Math.max(maxId, value.resource);
				}
			});
			maxId++;
			TransferableGraph1 tg1 = new TransferableGraph1(oldIdentities.size(), oldIdentities.toArray(new Identity[0]), new int[0], new Value[0]);
			boolean fixed = GraphRefactoringUtils.fixIncorrectRoot(tg1.identities);
			initInstructions(tg1, oldIdentities);
			if (fixed) {
				unfixIncorrectRoot(newIdentities);
			}
			oldIdentitiesCount = oldIdentities.size();
			oldIdentities.clear();
			
			init = true;
		}
		
		
		private void initInstructions(TransferableGraph1 tg1, List<Identity> oldIdentities) {
			BijectionMap<Identity, Identity> idMap = new BijectionMap<>();
			IdentityStore oldIdStore = TransferableGraphConversion.extractIdentities(tg1);
			
			newIdentities = new ArrayList<>();
			IdentityStore newIdStore = new IdentityStore();
			for (String s : oldIdStore.getRoots()) {
				int id = oldIdStore.pathToId(UriUtils.uriToPath(s));
				Identity oldRoot = getIdentity(oldIdentities, id);
				newIdStore.defineRoot(s, id);
				Identity identity = new Identity();
				identity.resource = id;
				identity.definition = new Root(s,((Root)oldRoot.definition).type);
				newIdentities.add(identity);
				idMap.map(getIdentity(oldIdentities, id), identity);
			}
			for (Entry<String, String> entry : identityMap.entrySet()) {
				Path from = UriUtils.uriToPath(entry.getKey());
				Path to = UriUtils.uriToPath(entry.getValue());
				
				int fromId = oldIdStore.pathToId(from);
				if (fromId < 0)
					continue; // We do not have the defined id in the file.
				int toId = oldIdStore.pathToId(to);
				if (toId > 0) {
					// new id is already included, we do not have to create a new identity.
					Identity id = getIdentity(oldIdentities, fromId);
					Identity id2 = getIdentity(oldIdentities, toId);
					idMap.map(id, id2);
					toSwap.put(fromId, toId);
				} else {
					Identity id = getIdentity(oldIdentities, fromId);
					
					// Find the shared ancestor for the new identity.
					List<Path> parentPaths = new ArrayList<>();
					Path current = to;
					while (true) {
						Path parent = ((PathChild)current).parent;
						int parentId = oldIdStore.pathToId(parent);
						parentPaths.add(0,parent);
						if (parentId > 0)
							break;
						current = parent;
					}
					Identity parentParent = getIdentity(oldIdentities,oldIdStore.pathToId(parentPaths.get(0)));
					for (int i = 1; i < parentPaths.size(); i++) {
						Path parent = parentPaths.get(i);
						String name = ((PathChild)parent).name;
						Identity pid = null;
						pid = getIdentity(newIdentities,newIdStore.pathToId(parentPaths.get(i)));
						if (pid == null) {
							// We need to copy shared ancestors from oldIdsStore to the newIDStore. Otherwise path queries
							// do not work.
							if (!newIdStore.hasIdentity(parentParent.resource)) {
								List<Identity> toCopy = new ArrayList<>();
								List<Path> toCopyPath = new ArrayList<>();
								Identity curr = parentParent;
								Path currPath = oldIdStore.idToPath(parentParent.resource);
								toCopy.add(curr);
								toCopyPath.add(currPath);
								while (true) {
									Path currParent = ((PathChild)toCopyPath.get(0)).parent;
									int parentId = newIdStore.pathToId(currParent);
									if (parentId > 0) {
										toCopy.add(0,getIdentity(newIdentities, parentId));
										toCopyPath.add(0,currParent);
										break;
									} else {
										toCopy.add(0,getIdentity(oldIdentities, parentId));
										toCopyPath.add(0,currParent);
									}
								}
								for (int j = 1; j < toCopy.size(); j++) {
									Identity _id = toCopy.get(i); 
									newIdentities.add(_id);
									int parId = newIdStore.pathToId(toCopyPath.get(i-1));
									newIdStore.defineChild(parId, ((External)_id.definition).name, _id.resource);
								}
							}
							pid = new Identity();
							pid.definition = new External(parentParent.resource, name);
							pid.resource = maxId++;
							newIdentities.add(pid);
							newIdStore.defineChild(parentParent.resource, name, pid.resource);
							parentParent = pid;
						}
					}
					
					Identity newId =  new Identity();
					newId.definition = new External(parentParent.resource, ((PathChild)to).name);
					newId.resource = maxId++;
					newIdentities.add(newId);
					idMap.map(id, newId);
					toSwap.put(id.resource, newId.resource);
				}
			}
			for (String uri : ignoreSet) {
				Path from = UriUtils.uriToPath(uri);
				int fromId = oldIdStore.pathToId(from);
				if (fromId < 0)
					continue; // We do not have the defined id in the file.
				toIgnore.add(fromId);
			}
			for (Identity oldId : oldIdentities) {
				if (!idMap.containsLeft(oldId) && !newIdentities.contains(oldId)) {
					if (toIgnore.contains(oldId.resource))
						continue;
					if (toSwap.containsKey(oldId.resource))
						continue;
					newIdentities.add(oldId);
				}
			}
		}
		
		private Identity getIdentity(List<Identity> identities, int id) {
			for (Identity identity : identities) {
				if (identity.resource == id)
					return identity;
			}
			return null;
		}
		
		int[] filter(int[] stm) {
			for (int i = 0; i < 4; i++) {
				if (toIgnore.contains(stm[i]))
					stm[i] = -1;
				else {
					Integer i2 = toSwap.get(stm[i]);
					if (i2 != null)
						stm[i] = i2;
				}
			}
			return stm;
		}
		
		@Override
		public int getIdentityCount() throws Exception {
			if (!init)
				return super.getIdentityCount();
			else {
				int c = super.getIdentityCount();
				// These are preparations of storing only external identities (instead of all).
				c -= oldIdentitiesCount;
				c += newIdentities.size();
				return c;
			}
		}
		
		@Override
		public void forIdentities(ReadGraph graph, TransferableGraphSourceProcedure<Identity> procedure)
				throws Exception {
			for (Identity id : newIdentities) {
				procedure.execute(id);
			}
		}
		
		@Override
		public void forStatements(ReadGraph graph, TransferableGraphSourceProcedure<int[]> procedure) throws Exception {
			int[] value = new int[4];

			int stmLength = getStatementCount();

			for(int stmIndex=0;stmIndex<stmLength;) {

				value[stmIndex & 3] = safeInt();
				stmIndex++;
				if((stmIndex & 3) == 0) procedure.execute(value);

				// Cached bytes 
				int avail = (SIZE-byteIndex) >> 2;
				int allowed = Math.min(stmLength-stmIndex, avail);
				for(int index = byteIndex, i=0;i<allowed;i++) {
					value[stmIndex & 3] = ((bytes[index++]&0xff)<<24) | ((bytes[index++]&0xff)<<16) | ((bytes[index++]&0xff)<<8) | ((bytes[index++]&0xff));
					stmIndex++;
					if((stmIndex & 3) == 0) {
						value = filter(value);
						if (value[0] < 0 || value[3] < 0)
							continue;
						if (value[1] >= 0) {
							procedure.execute(value);	
						} else if (value[1] < 0 && value[2] >= 0) {
							int t = value[0];
							value[0] = value[3];
							value[3] = t;
							t = value[1];
							value[1] = value[2];
							value[2] = t;
							procedure.execute(value);
						}
					}			
				}
				byteIndex += allowed<<2;

			}
		}
		
		@Override
		public void forValues(ReadGraph graph, TransferableGraphSourceProcedure<Value> procedure) throws Exception {
			super.forValues(graph, procedure);
		}
		
		@Override
		public void forValues2(ReadGraph graph, TransferableGraphSourceValueProcedure procedure) throws Exception {
			super.forValues2(graph, procedure);
		}
		
	}

}
