/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.g2d.element.handler.impl;

import org.simantics.g2d.element.ElementUtils;
import org.simantics.g2d.element.IElement;
import org.simantics.g2d.element.handler.TextEditor;
import org.simantics.g2d.utils.TextSegment;
import org.simantics.utils.datastructures.hints.IHintContext.Key;
import org.simantics.utils.datastructures.hints.IHintContext.KeyOf;

public class TextEditorImpl implements TextEditor {

    private static final long      serialVersionUID = 7183657432363119779L;

    public static final TextEditor INSTANCE         = new TextEditorImpl();

    public static final Key        ACTIVE           = new KeyOf(Boolean.class, "TEXT_EDITOR_ACTIVE");
    public static final Key        TEXT             = new KeyOf(String.class, "TEXT_EDIT_TEXT");
    public static final Key        MODIFIER         = new KeyOf(Modifier.class, "TEXT_MODIFIER");
    public static final Key        TEXT_SELECTION   = new KeyOf(TextSegment.class, "TEXT_SELECTION");
    public static final Key        CARET_POS        = new KeyOf(Integer.class, "CARET_POS");

    @Override
    public boolean isActive(IElement e) {
        return ElementUtils.getHintOrDefault(e, ACTIVE, Boolean.FALSE);
    }

    @Override
    public void setActive(IElement e, boolean value) {
        ElementUtils.setOrRemoveHint(e, ACTIVE, value ? Boolean.TRUE : null);
        if (!value) {
            e.removeHint(CARET_POS);
        }
    }

    @Override
    public String getText(IElement e) {
        return e.getHint(TEXT);
    }

    @Override
    public void setText(IElement e, String text) {
        ElementUtils.setOrRemoveHint(e, TEXT, text);
    }

    @Override
    public Modifier getModifier(IElement e) {
        return e.getHint(MODIFIER);
    }

    @Override
    public void setModifier(IElement e, org.simantics.g2d.element.handler.TextEditor.Modifier modifier) {
        ElementUtils.setOrRemoveHint(e, MODIFIER, modifier);
    }

    @Override
    public Integer getCaretPosition(IElement e) {
        Integer pos = e.getHint(CARET_POS);
        return pos;
    }

    @Override
    public TextSegment getSelection(IElement e) {
        return e.getHint(TEXT_SELECTION);
    }

    private static boolean isValidCaretPosition(String s, int pos) {
        int l = s.length();
        return pos >= 0 && pos <= l; 
    }

    private static void assertValidCaretPosition(String s, int pos) {
        boolean valid = isValidCaretPosition(s, pos);
        if (!valid)
            throw new IllegalArgumentException("caret position " + pos + " is out of bounds in text '" + s + "'");
    }

    @Override
    public void setCaretPosition(IElement e, Integer position) {
        if (position == null) {
            e.removeHint(CARET_POS);
        } else {
            String text = getText(e);
            Integer old = e.getHint(CARET_POS);
            if (position.equals(old))
                return;
            assertValidCaretPosition(text, position);
            e.setHint(CARET_POS, position);
        }
    }

    @Override
    public void setSelection(IElement e, TextSegment segment) {
        if (segment == null) {
            e.removeHint(TEXT_SELECTION);
        } else {
            String text = getText(e);
            TextSegment old = e.getHint(TEXT_SELECTION);
            if (segment.equals(old))
                return;
            if (!segment.existsIn(text))
                throw new IllegalArgumentException("segment " + segment + " does not exist in text '" + text + "'");
            e.setHint(TEXT_SELECTION, segment);
        }
    }

}
