/*******************************************************************************
 * Copyright (c) 2007, 2023 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - GitLab #1015
 *******************************************************************************/
package org.simantics.g2d.diagram.participant.pointertool;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.HashSet;

import org.simantics.g2d.canvas.impl.DependencyReflection.Dependency;
import org.simantics.g2d.canvas.impl.DependencyReflection.Reference;
import org.simantics.g2d.canvas.impl.SGNodeReflection.SGCleanup;
import org.simantics.g2d.canvas.impl.SGNodeReflection.SGInit;
import org.simantics.g2d.diagram.DiagramHints;
import org.simantics.g2d.diagram.handler.PickContext;
import org.simantics.g2d.diagram.handler.PickRequest;
import org.simantics.g2d.diagram.handler.PickRequest.PickPolicy;
import org.simantics.g2d.diagram.participant.Selection;
import org.simantics.g2d.element.IElement;
import org.simantics.g2d.participant.KeyUtil;
import org.simantics.g2d.participant.RenderingQualityInteractor;
import org.simantics.g2d.participant.TransformUtil;
import org.simantics.scenegraph.g2d.G2DParentNode;
import org.simantics.scenegraph.g2d.events.Event;
import org.simantics.scenegraph.g2d.events.EventHandlerReflection.EventHandler;
import org.simantics.scenegraph.g2d.events.MouseEvent;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseWheelMovedEvent;
import org.simantics.scenegraph.g2d.events.command.CommandEvent;
import org.simantics.scenegraph.g2d.events.command.Commands;
import org.simantics.scenegraph.g2d.nodes.BoxSelectionNode;
import org.simantics.scenegraph.g2d.nodes.BoxSelectionStrategy;
import org.simantics.scenegraph.g2d.nodes.SelectionPolicy;
import org.simantics.scenegraph.utils.Quality;

/**
 * @author Toni Kalajainen
 */
public class BoxSelectionMode extends AbstractMode {

    @Reference  RenderingQualityInteractor quality;
    @Dependency TransformUtil util;
    @Dependency Selection selection;
    @Dependency PickContext pickContext;
    @Dependency KeyUtil keyUtil;

    public final static BasicStroke STROKE         = new BasicStroke(1.0f);

    public static final int         PAINT_PRIORITY = 30;

    Point2D                         startingPoint;
    Point2D                         currentPoint;
    int                             mouseButton;
    PickPolicy                      boxSelectMode;
    BoxSelectionNode                node;
    BoxSelectionStrategy            boxSelectStrategy;

    private static PickPolicy convert(SelectionPolicy p, PickPolicy defaultValue) {
        if (p == null)
            return defaultValue;
        switch (p) {
        case SELECT_INTERSECTING: return PickPolicy.PICK_INTERSECTING_OBJECTS;
        case SELECT_CONTAINED: default: return PickPolicy.PICK_CONTAINED_OBJECTS;
        }
    }

    protected PickPolicy choosePickPolicy(PickPolicy defaultValue) {
        PickPolicy p = boxSelectMode;
        if (boxSelectStrategy != null) {
            p = convert(
                    boxSelectStrategy.chooseSelectionPolicy(node.getStart(), node.getEnd()),
                    boxSelectMode);
        }
        return p != null ? p : defaultValue;
    }

    @SGInit
    public void init(G2DParentNode parent) {
        if (quality != null)
            quality.setStaticQuality(Quality.LOW);

        node = parent.getOrCreateNode("" + hashCode(), BoxSelectionNode.class);
        node.setZIndex(PAINT_PRIORITY);
        node.setStart(startingPoint);
        node.setEnd(currentPoint);
        node.setStroke(STROKE);
        node.setScaleStroke(true);
        node.setColor(getToolColor());
        node.setStrategy(boxSelectStrategy);
        node.setMouseButton(mouseButton);
        node.setSelectionListener(new BoxSelectionNode.SelectionListener() {
            @Override
            public void onSelect(Rectangle2D rect, int modifiers) {
                // Disposed?
                if (isRemoved())
                    return;

                boolean toggle = (modifiers & MouseEvent.CTRL_MASK) != 0;
                boolean accumulate = (modifiers & MouseEvent.SHIFT_MASK) != 0;

                var boxSelection = new HashSet<IElement>();
                PickRequest request = new PickRequest(rect).context(getContext());
                request.pickPolicy = choosePickPolicy(request.pickPolicy);
                pickContext.pick(diagram, request, boxSelection);

                int selectionId = mouseId;
                if (toggle) {
                    selection.toggle(selectionId, boxSelection);
                } else if (accumulate) {
                    for (IElement elem : boxSelection)
                        selection.add(selectionId, elem);
                } else {
                    selection.setSelection(selectionId, boxSelection);
                }

                if (node != null) {
                    node.remove();
                    node = null;
                }

                setDirty();
                remove();
            }
        });

    }

    @SGCleanup
    public void cleanup() {
        if (quality != null)
            quality.setStaticQuality(null);
        if (node != null) {
            node.remove();
            node = null;
        }
    }

    public BoxSelectionMode(Point2D startingPoint, Point2D currentPoint, int mouseId, int mouseButton, PickPolicy boxSelectMode) {
        super(mouseId);
        this.startingPoint = startingPoint;
        this.currentPoint = currentPoint;
        this.mouseButton = mouseButton;
        this.boxSelectMode = boxSelectMode;
        //setHint(RenderingQualityInteractor.KEY_QUALITY_INTERACTOR_ENABLED, Boolean.FALSE);
    }

    public BoxSelectionMode(Point2D startingPoint, Point2D currentPoint, int mouseId, int mouseButton, PickPolicy boxSelectMode, BoxSelectionStrategy boxSelectStrategy) {
        this(startingPoint, currentPoint, mouseId, mouseButton, boxSelectMode);
        this.boxSelectStrategy = boxSelectStrategy;
    }

    public synchronized Color getToolColor() {
        Color c = getHint(DiagramHints.KEY_SELECTION_FRAME_COLOR);
        if (c != null)
            return c;
        return Color.BLACK;
    }

    /**
     * Allows user to cancel box selection operations by invoking the CANCEL
     * command (pressing ESC).
     */
    @EventHandler(priority = 100)
    public boolean handleCancel(CommandEvent e) {
        if (e.command.equals( Commands.CANCEL ) ) {
            setDirty();
            remove();
            return true;
        }
        return false;
    }
    
    @EventHandler(priority = Integer.MAX_VALUE)
    public boolean handleEvent(Event e) {
    	return node.handleEvent(e) || ((e instanceof CommandEvent) && handleCancel((CommandEvent)e)) || (!(e instanceof MouseWheelMovedEvent));
    }
    

}