/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.g2d.connection;

import org.simantics.g2d.diagram.handler.Topology.Terminal;
import org.simantics.g2d.element.handler.EdgeVisuals.EdgeEnd;
import org.simantics.utils.datastructures.hints.IHintContext.KeyOf;

/**
 * This hint key is used for representing a terminal within an element that is
 * connected to an edge. The key is identified by two things: the connected
 * terminal and a secondary data object that can be used to make the key unique
 * for an edge in cases where several connections to the same terminal are
 * possible. A good option is to use the back-end object of the edge element
 * itself, which naturally makes the key unique for that particular edge.
 * 
 * <p>
 * This key has a counterpart, {@link EndKeyOf} that is used in edge elements to
 * connect both the {@link EdgeEnd#Begin} and {@link EdgeEnd#End} ends of the
 * edge to element terminals. To actually represent the connection, you should
 * give the same value object for this hint pair.
 * </p>
 * 
 * @author Tuukka Lehtonen
 */
public class TerminalKeyOf extends KeyOf {

    private final Terminal terminal;
    private final Object   data;
    private int            hash;

    /**
     * @param t the terminal this connector hint represents
     * @param data a secondary data object for making the key unique for a
     *        single edge connected to the specified terminal. Use a
     *        <code>null</code> value only if you consider the terminal to allow
     *        only one connected edge.
     * @param clazz the allowed class of the value object.
     */
    public TerminalKeyOf(Terminal t, Object data, Class<?> clazz) {
        super(clazz);
        this.terminal = t;
        this.data = data;
        this.hash = t.hashCode() ^ clazz.hashCode();
        if (data != null)
            this.hash = this.hash * 31 + data.hashCode();
    }

    public Terminal getTerminal() {
        return terminal;
    }

    public Object getData() {
        return data;
    }

    @Override
    public int hashCode() {
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (!(obj instanceof TerminalKeyOf))
            return false;
        TerminalKeyOf other = (TerminalKeyOf) obj;
        if (!terminal.equals(other.terminal))
            return false;
        return (data == null) ? other.data == null : data.equals(other.data);
    }

    @Override
    public String toString() {
        return getClass().getSimpleName() + "[terminal=" + terminal + ", data=" + data + "]";
    }

}