package org.simantics.export.ui;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.SubMonitor;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.IWizardPage;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.osgi.util.NLS;
import org.eclipse.ui.IExportWizard;
import org.eclipse.ui.IWorkbench;
import org.osgi.service.prefs.BackingStoreException;
import org.simantics.Simantics;
import org.simantics.db.exception.DatabaseException;
import org.simantics.export.core.ExportContext;
import org.simantics.export.core.Exports;
import org.simantics.export.core.error.ExportException;
import org.simantics.export.core.manager.ExportManager;
import org.simantics.export.core.manager.ExportPlan;
import org.simantics.export.core.manager.ExportWizardResult;
import org.simantics.export.core.util.ExporterUtils;
import org.simantics.utils.datastructures.collections.CollectionUtils;
import org.simantics.utils.ui.ErrorLogger;
import org.simantics.utils.ui.ExceptionUtils;

public class ExportCoreWizard extends Wizard implements IExportWizard {

	ExportContext ctx;
	List<String> selection;

	ContentSelectionPage contentPage;
	OptionsPage optionsPage;

	public ExportCoreWizard() {
		setWindowTitle(Messages.ExportCoreWizard_ExportPDFFiles);
		setNeedsProgressMonitor(true);
	}

	public void init(IWorkbench workbench, final IStructuredSelection selection) {
		try {
			// Create export context
			ctx = ExportContext.create( Simantics.getSessionContext(), selection );

			// Create extension point registry
			ctx.eep = Exports.createExtensionPoint();

		} catch (DatabaseException e) {
			ExceptionUtils.logAndShowError(e);
		}
	}

	@Override
	public boolean performFinish() {

		// User clicked finish on the first page. Preapare options page.
		if ( getContainer().getCurrentPage() == contentPage ) {
			contentPage.validatePage();
			optionsPage.update( contentPage.getContentSelection() );
		}

		final boolean[] canceled = new boolean[1];

		try {
			contentPage.savePrefs();
			optionsPage.savePrefs();
		} catch (ExportException e) {
			e.printStackTrace();
			ExceptionUtils.logError(e);
		}

		try {
			final ExportPlan plan = new ExportPlan();
			final ExportWizardResult wizardResult = optionsPage.getOutput();
			final ExportManager em = new ExportManager( wizardResult.options, ctx );
			wizardResult.createPlan( ctx, plan );
			System.out.println(wizardResult);

			List<String> exportProblems = em.validate(ctx, plan);
			if ( !exportProblems.isEmpty() ) {
				CollectionUtils.unique(exportProblems);
				WizardPage cp = (WizardPage) getContainer().getCurrentPage();
				String str = CollectionUtils.toString(exportProblems, "\n"); //$NON-NLS-1$
				cp.setErrorMessage( str );
				return false;
			}

			getContainer().run(true, true, new IRunnableWithProgress() {
				@Override
				public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
					SubMonitor mon = SubMonitor.convert(monitor, plan.label, 1000000);
					try {
						em.execute(ctx, mon.newChild(1000000, SubMonitor.SUPPRESS_NONE), plan);
					} catch (Exception e) {
						throw new InvocationTargetException(e);
					} finally {
						canceled[0] = monitor.isCanceled();
						monitor.done();
					}
				}
			});

			try {
				ctx.store.flush();
			} catch (BackingStoreException e) {
				ErrorLogger.defaultLogError(Messages.ExportCoreWizard_FailedToSavePreferences, e);
				ExceptionUtils.logError(e);
			}
		} catch (InvocationTargetException e) {
			Throwable t = e.getTargetException();
			WizardPage cp = (WizardPage) getContainer().getCurrentPage();

			if ( t instanceof ExportException && t.getCause()!=null) {
				ExportException ee = (ExportException) t;
				if ( ee.getCause() != null ) t = ee.getCause();
			}

			if (canceled[0]) {
				cp.setErrorMessage(Messages.ExportCoreWizard_FailedToPersistWizardPrefs);
			} else if (t instanceof IOException) {
				ErrorLogger.defaultLogError("An I/O problem occurred while exporting the model. See exception for details.", t); //$NON-NLS-1$
				cp.setErrorMessage(NLS.bind(Messages.ExportCoreWizard_IOProblem, t.getMessage()));
			} else {
				ErrorLogger.defaultLogError("Unexpected exception while exporting the model. See exception for details.", t); //$NON-NLS-1$
				cp.setErrorMessage(NLS.bind(Messages.ExportCoreWizard_UnexpectedException, t.getMessage()));
			}
			return false;
		} catch (InterruptedException e) {
			ExceptionUtils.logAndShowError(e);
		} catch (ExportException e) {
			ExceptionUtils.logAndShowError(e);
		}

		return true;
	}

	@Override
	public void addPages() {
		super.addPages();
		try {
			contentPage = new ContentSelectionPage(ctx);
			optionsPage = new OptionsPage(ctx);
			addPage( contentPage );
			addPage( optionsPage );
		} catch (ExportException e) {
			e.printStackTrace();
			ExceptionUtils.logError(e);			
		}
	}

	@Override
	public IWizardPage getNextPage(IWizardPage page) {
		if ( page == optionsPage ) {
			try {
				optionsPage.getOutput();
			} catch (ExportException e) {
				return null;
			}
		}

		if ( page == contentPage ) {
			contentPage.validatePage();
			optionsPage.update( ExporterUtils.sortContent( contentPage.getContentSelection() ) );
		}
		return super.getNextPage(page);
	}

}
