/*******************************************************************************
 * Copyright (c) 2017 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.export.core.pdf;

import java.util.concurrent.atomic.AtomicBoolean;

import com.lowagie.text.FontFactory;
import com.lowagie.text.pdf.DefaultFontMapper;
import com.lowagie.text.pdf.FontMapper;

/**
 * A utility class for constructing properly initialize iText {@link FontMapper}
 * instances.
 * 
 * @author Tuukka Lehtonen
 * @since 1.28.0
 */
public class FontMapping {

	private static final AtomicBoolean fontFactoryInitialized = new AtomicBoolean();

	public static FontMapper defaultFontMapper() {
		// NOTE: recreation is intentional because users might add fonts between exports.
		// At some point, perhaps we could listen to file system changes in these locations
		// and only then recreate the mapper.
		return createDefaultFontMapper(); 
	}

	public static FontMapper createDefaultFontMapper() {
		initializeFontFactory();
		DefaultFontMapper mapper = new DefaultFontMapper();
		insertDirectories(mapper);
		return mapper;
	}

	private static void insertDirectories(DefaultFontMapper mapper) {
		switch (OSType.calculate()) {
		case APPLE:
			mapper.insertDirectory("/Library/Fonts");
			mapper.insertDirectory("/System/Library/Fonts");

		case LINUX:
		case SUN:
			mapper.insertDirectory("/usr/share/X11/fonts");
			mapper.insertDirectory("/usr/X/lib/X11/fonts");
			mapper.insertDirectory("/usr/openwin/lib/X11/fonts");
			mapper.insertDirectory("/usr/share/fonts");
			mapper.insertDirectory("/usr/X11R6/lib/X11/fonts");
			break;

		case WINDOWS:
			String windir = System.getenv("WINDIR");
			if (windir != null && !windir.isEmpty()) {
				mapper.insertDirectory(windir + "\\Fonts");
			}
			break;

		default:
			break;
		}
	}

	private static void initializeFontFactory() {
		if (!fontFactoryInitialized.compareAndSet(false, true))
			return;

		switch (OSType.calculate()) {
		case APPLE:
			FontFactory.registerDirectory("/Library/Fonts");
			FontFactory.registerDirectory("/System/Library/Fonts");

		case LINUX:
		case SUN:
			FontFactory.registerDirectory("/usr/share/X11/fonts", true);
			FontFactory.registerDirectory("/usr/X/lib/X11/fonts", true);
			FontFactory.registerDirectory("/usr/openwin/lib/X11/fonts", true);
			FontFactory.registerDirectory("/usr/share/fonts", true);
			FontFactory.registerDirectory("/usr/X11R6/lib/X11/fonts", true);
			break;

		case WINDOWS:
			String windir = System.getenv("WINDIR");
			if (windir != null && !windir.isEmpty()) {
				FontFactory.registerDirectory(windir + "\\Fonts");
			}
			break;

		default:
			break;
		}
	}

	static enum OSType {
		APPLE, LINUX, SUN, WINDOWS, UNKNOWN;

		public static OSType calculate() {
			String osName = System.getProperty("os.name");
			assert osName != null;
			osName = osName.toLowerCase();
			if (osName.startsWith("mac os x"))
				return APPLE;
			if (osName.startsWith("windows"))
				return WINDOWS;
			if (osName.startsWith("linux"))
				return LINUX;
			if (osName.startsWith("sun"))
				return SUN;
			return UNKNOWN;
		}
	}

}
